package controllers;

import play.Logger;
import play.Play;
import play.data.validation.Required;
import play.libs.WS;
import play.libs.WS.HttpResponse;
import play.mvc.Before;
import play.mvc.Controller;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;


public class BrowserIDSecure extends Controller {

	private static final String BROWSERID_ENFORCESSL = "browserid.enforcessl";
	private static final String BROWSERID_AUDIENCE = "browserid.audience";
	private static final String BROWSERID_LOGOUT_REDIRECT = "browserid.logoutRedirect";
	private static final String ORIGINAL_URL = "originalUrl";
	private static final String USER_COOKIE = "bid.user";

	public static boolean requiresSSL=false;
	
	static {
	
		String enforceSSL = Play.configuration.getProperty(
				BROWSERID_ENFORCESSL, "false");
		Boolean b = new Boolean(enforceSSL);
		requiresSSL=b.booleanValue();
	
	}


	@Before(unless = { "login", "authenticate", "logout" })
	static void checkAccess() throws Throwable {

		final String userId = getUserId();

		if (userId == null) {
			final String originalUrl = request.method.equals("GET") ? request.url
					: "/";
			flash.put(ORIGINAL_URL, originalUrl);
			login();
		}
	}

	/**
	 * The action for the login page mapped to GET /login
	 * Shows the login page
	 */
	public static void login() {
		flash.keep(ORIGINAL_URL);
		render();
	}

	/**
	 * The logout action mapped to /logout
	 *
	 * Logs the user out by deleting the USER_COOKIE
	 */
	public static void logout() {
		session.remove(USER_COOKIE);
		final String redirectTo = Play.configuration.getProperty(
				BROWSERID_LOGOUT_REDIRECT, "/");
		redirect(redirectTo);
	}

	/**
	 * Action mapped to a POST /login
	 * 
	 * Calls browserid's rest api via https to validate the authenticity token
	 * Reports status accordinly
	 *
     */
	public static void authenticate(@Required String assertion,
			String authenticityToken) {

		String originalUrl = null;

		if (validation.hasErrors()) {
			Logger.error("Assertion is required");
			notFound();
		}
		flash.keep(ORIGINAL_URL);

		// Cross site forgery check
		checkAuthenticity();

		// Read the audience from configuration rather than the request
		// that's a recommendation from BrowserID
		String audience = Play.configuration.getProperty(BROWSERID_AUDIENCE);
		if (audience == null) {
			String msg = "In application.conf, please set property browserid.audience to your site url, typically http://www.mydomain.com";
			Logger.error(msg);
			forbidden(msg);
		}
		
		if (requiresSSL && request.secure == false) {
			String msg = "Not an SSL connection";
			Logger.error(msg);
			forbidden(msg);
		}

		HttpResponse res = WS.url("https://browserid.org/verify")
				.setParameter("assertion", assertion)
				.setParameter("audience", audience).post();

		int status = res.getStatus();

		if (status == 200) {
			JsonElement json = res.getJson();
			if (json.isJsonObject()) {
				JsonObject o = (JsonObject) json;
				JsonElement v = o.get("status");
				if (v != null && "okay".equals(v.getAsString())) {
					String email = o.get("email").getAsString();

					// Do we really need this ?
					// We may want to check the issuer and the validity
					// String validity = o.get("valid-until").getAsString();
					// String issuer = o.get("issuer").getAsString();
					setUserId(email);

					originalUrl = flash.get(ORIGINAL_URL);
					redirect(originalUrl != null ? originalUrl : "/");

				} else
					Logger.error(o.get("reason").getAsString());
			}
		} else
			Logger.error("BAD STATUS =" + status);

		flash.keep(ORIGINAL_URL);
		login();
	}

	/**
	 * Returns true if there is a user logged in or false otherwise.
	 * 
	 * @return a boolean
	 */
	public static boolean isUserLoggedIn() {
		return getUserId() != null;
	}

	/**
	 * Sets the SecureSocial cookies in the session.
	 */
	private static void setUserId(String user) {
		session.put(USER_COOKIE, user);
	}

	/**
	 * Creates a UserId object from the values stored in the session.
	 * 
	 * @returns the user id
	 */
	private static String getUserId() {
		return session.get(USER_COOKIE);
	}
}
