package play.modules.cnm;

import java.io.IOException;
import java.util.Properties;

import play.Logger;
import play.Play;
import play.PlayPlugin;
import play.libs.IO;
import play.mvc.Http.Request;

/**
 * <h1>Overview</h1>
 * <p>The content negotiation plugin is used to map content types found in the accept header of a HTTP request
 * to the corresponding response format.</p>
 * <p>By default the plugin uses its own content types file to map content types and response format but you
 * can overwrite this file by specifying your content types mapping using the {@link #CONTENT_TYPES_FILE}
 * key.</p>
 *
 * <h1>Caveats</h1>
 * <ul>
 * 	<li>This plugin is only useful when clients do send <b>no more then one</b> content type in their accept header.
 *      Sending more then one acceptable content type will not match any specified content type with this plugin
 *      and because of that the default Play! content negotiation will be used. This is necessary because right now
 *      there is no way of knowing whether a given response format is available or not before invoking the action
 *      and therefore do proper content negotiation.</li>
 * </ul>
 *
 * <h1>Examples</h1>
 * <ol>
 * 	<li>
 *      <p>Enabling custom content type mappings</p>
 *      <p>If you want to enable a custom content type like:
 *      <pre>   application/vnd.example+xml</pre>
 *      to the format <code>.example</code> you can write the following into a text file:
 *      <pre>   application/vnd.example+xml=example</pre>
 *      That way any request whose accept header is set to <code>application/vnd.example</code> will be answered
 *      by templates ending with <code>.example</code></p>
 *      <p>To enable the mapping you have to specify the path of your text file inside your 
 *      <code>application/conf</code> file using the {@linkplain #CONTENT_TYPES_FILE} key:
 *      <pre>   content.types=conf/types</pre></p>
 *  </li>
 * </ol>
 *
 * <h1>How to help</h1>
 * <ul>
 *  <li>Test the plugin and write back about errors, bugs and wishes.</li>
 *  <li>Find out how Play! currently handles accept headers.</li>
 *  <li>Check whether we can hook into the current content negotiation system.</li>
 *  <li>Alternatively check if it is possible to extend Play! in a way to set multiple response formats and Play! 
 *      determines the best possible match itself.</li>
 *  <li>Try to save the content type mappings file in the public folder. Maybe it is useful for clients to know
 *      the list of available content types?</li>
 * </ul>
 *
 * @author  Sebastian Hoß (mail@shoss.de)
 * @since   1.0
 * @version 1.0
 */
public final class ContentNegotiationPlugin extends PlayPlugin {

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    // *                                            CONSTANTS                                            *
    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    /** The configuration key used the set the content types file. */
    public static final String  CONTENT_TYPES_FILE          = "content.types"; //$NON-NLS-1$

    /** The path to the default content types file. */
    private static final String DEFAULT_CONTENT_TYPES_FILE  = "conf/types"; //$NON-NLS-1$

    /** The name of the accept header in any given {@link Request request}. */
    private static final String ACCEPT_HEADER               = "accept"; //$NON-NLS-1$

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    // *                                            ATTRIBUTES                                           *
    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    /** The current map of allowed content types and their corresponding formats. */
    private Properties          contentTypes;

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    // *                                             METHODS                                             *
    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    /**
     * <p>Loads either the default or a user-defined content type mappings file (a.k.a. properties file).</p>
     */
    @Override
    public void onConfigurationRead() {
        // Set correct path
        String path = Play.configuration.getProperty(CONTENT_TYPES_FILE) != null ? 
                Play.configuration.getProperty(CONTENT_TYPES_FILE) : DEFAULT_CONTENT_TYPES_FILE;

        try {
            // load properties file from path
            this.contentTypes = IO.readUtf8Properties(Play.getVirtualFile(path).inputstream());
        } catch (RuntimeException exception) {
            if (exception.getCause() instanceof IOException) {
                Logger.fatal("Cannot read allowed content types");
                System.exit(0);
            }
        }
    }

    /**
     * <p>Sets the request format using the {@link #contentTypes content types mapping}.</p>
     */
    @Override
    public void beforeInvocation() {
        if (this.shouldIntercept()) {
            // Set the request format to the corresponding value for the current accept header
            Request.current().format = this.contentTypes.getProperty(Request.current().headers.get(ACCEPT_HEADER).value());
        }
    }

    /**
     * <p>Uses the previously set {@link #contentTypes content types} to check whether this plugin should interfere
     * with the request or not. If no matching content type is found no changes are going to happen. That way we 
     * won't brake any behavior with the existing Play! content type matcher.</p> 
     *
     * TODO: Which Play! class is responsible for content type negotiation ATM?
     *
     * @return	<code>true</code> if an accept header contains any known value, <code>false</code> otherwise.
     */
    private boolean shouldIntercept() {
        // Check if the request contains an accept header
        if (Request.current().headers.containsKey(ACCEPT_HEADER)) {
            // If the request contains an accept header..

            // ..check whether the accept header is a known value
            if (this.contentTypes.containsKey(Request.current().headers.get(ACCEPT_HEADER).value())) {
                // If the request is a known value..

                // ..approve!
                return true;
            }
        }
        // If the request does not contain an accept header or the value is not known..

        // ..deny!
        return false;
    }

}
