/*
 * Project: play-content-negotiation
 * Package: play.modules.cnm
 * File   : ContentNegotiationPlugin.java
 * Created: 17.10.2010 - 18:03:34
 *
 *
 * Copyright 2010 Sebastian Hoß
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package play.modules.cnm;

import java.lang.annotation.Inherited;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import play.PlayPlugin;
import play.mvc.Http.Header;
import play.mvc.Http.Request;

/**
 * <h1>Overview</h1>
 * <p>The content negotiation plugin is used to map content types found in the accept header of a HTTP request
 * to the corresponding response format.</p>
 *
 * <h1>Caveats</h1>
 * <ul>
 * 	<li>Nothing so far.</li>
 * </ul>
 *
 * <h1>How to help</h1>
 * <ul>
 *  <li>Test the plugin and write back about errors, bugs and wishes.</li>
 * </ul>
 *
 * @author  Sebastian Hoß (mail@shoss.de)
 * @since   1.0
 * @see     Supports
 * @see     MediaType
 */
public final class ContentNegotiationPlugin extends PlayPlugin {

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    // *                                            CONSTANTS                                            *
    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    /** The name of the accept header in any given {@link Request request}. */
    private static final String ACCEPT_HEADER               = "accept"; //$NON-NLS-1$

    /** The RegEx used to differentiate between the name and the value of elements inside the accept header. */
    private static final String ACCEPT_PATTERN              = "q=([0-9\\.]+)"; //$NON-NLS-1$

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    // *                                            ATTRIBUTES                                           *
    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    /**
     * <p>Sets the request format using the Produces annotation either found on the action itself or the enclosing class.</p>
     */
    @Override
    public void beforeActionInvocation(final Method actionMethod) {
        // Check whether the given method supports custom media types and the client did send an accept header
        if (this.supportsCustomType(actionMethod) && this.hasMatch(actionMethod, Request.current().headers)) {
            // If custom media types are supported and the given method matches a value from the accept header..

            // ..set the request format to the corresponding value for the current accept header
            Request.current().format = this.findBestMatch(actionMethod, Request.current().headers.get(ACCEPT_HEADER).value());
        }
        // Custom media types are not supported or the given method matches no value from the accept header..

        // ..do nothing and let Play! handle further content negotiation
    }

    /**
     * <p>
     * Checks whether a given {@link Method method} supports a custom {@link MediaType}. For that it looks at the given method
     * and the declaring class of that method. Since the {@link Supports} annotation is marked as {@link Inherited} we can be
     * sure to work with the entire hierarchy of the declaring class.
     * </p>
     *
     * @param actionMethod  The method to check (<b>may not be <code>null</code></b>).
     * @return              <code>true</code> if the given methods supports a custom media type, <code>false</code> otherwise.
     */
    private boolean supportsCustomType(final Method actionMethod) {
        return (actionMethod.isAnnotationPresent(Supports.class) ||
                actionMethod.getDeclaringClass().isAnnotationPresent(Supports.class)) &&
                ((actionMethod.getAnnotation(Supports.class).value().length > 0) ||
                        (actionMethod.getDeclaringClass().getAnnotation(Supports.class).value().length > 0));
    }

    /**
     * <p>
     * Checks whether there is a match for a given header and a given method.
     * </p>
     *
     * @param actionMethod  The method to check (<b>may not be <code>null</code></b>).
     * @param headers       The headers to check (<b>may not be <code>null</code></b>).
     * @return              <code>true</code> if any match was found, <code>false</code> otherwise.
     */
    private boolean hasMatch(final Method actionMethod, final Map<String, Header> headers) {
        return headers.containsKey(ACCEPT_HEADER) && !headers.get(ACCEPT_HEADER).toString().isEmpty() &&
        (this.findBestMatch(actionMethod, Request.current().headers.get(ACCEPT_HEADER).value()) != null);
    }

    /**
     * <p>
     * Finds the best matching {@link MediaType} for a given accept header and a given method.
     * </p>
     *
     * @param actionMethod  The method to use (<b>may not be <code>null</code></b>).
     * @param header        The header to use (<b>may not be <code>null</code></b>).
     * @return              The accept header value of the best matching media type.
     */
    private String findBestMatch(final Method actionMethod, final String header) {
        // Get the supported media types
        final Map<String, String> supportedTypes = this.determineSupportedTypes(actionMethod);

        // Get and rank the accepted media types
        final List<String> acceptedTypes = this.sortValues(Arrays.asList(header.split(","))); //$NON-NLS-1$

        // Check whether the accept header contains a known, custom value
        for (final String type : acceptedTypes) {
            if (supportedTypes.containsKey(type)) {
                // If the request is a known value..

                // ..approve!
                return supportedTypes.get(type);
            }
        }

        // If the request is not a known value
        return null;
    }

    /**
     * <p>
     * Builds a map of supported media types for a given method.
     * </p>
     *
     * @param actionMethod  The method to use (<b>may not be <code>null</code></b>).
     * @return              A map containing all supported media types by either the method itself or the declaring class.
     */
    private Map<String, String> determineSupportedTypes(final Method actionMethod) {
        final Map<String, String> supportedTypes = new HashMap<String, String>();

        final List<MediaType> methodTypes = new ArrayList<MediaType>();
        final List<MediaType> classTypes =  new ArrayList<MediaType>();

        if (actionMethod.isAnnotationPresent(Supports.class)) {
            methodTypes.addAll(Arrays.asList(actionMethod.getAnnotation(Supports.class).value()));
        }

        if (actionMethod.getDeclaringClass().isAnnotationPresent(Supports.class)) {
            classTypes.addAll(Arrays.asList(actionMethod.getDeclaringClass().getAnnotation(Supports.class).value()));
        }


        for (final MediaType type : methodTypes) {
            supportedTypes.put(type.accept().trim(), type.format().trim());
        }

        for (final MediaType type : classTypes) {
            supportedTypes.put(type.accept().trim(), type.format().trim());
        }


        return supportedTypes;
    }

    /**
     * <p>
     * Sorts values based on their q-value.
     * </p>
     *
     * @param unsortedTypes The list of unsorted values.
     * @return  The sorted list of values, without their q-value.
     */
    private List<String> sortValues(final List<String> unsortedTypes) {
        final Pattern qpattern = Pattern.compile(ACCEPT_PATTERN);

        Collections.sort(unsortedTypes, new Comparator<String>() {
            public int compare(final String first, final String second) {
                double q1 = 1.0;
                double q2 = 1.0;
                final Matcher m1 = qpattern.matcher(first);
                final Matcher m2 = qpattern.matcher(second);
                if (m1.find()) {
                    q1 = Double.parseDouble(m1.group(1));
                }
                if (m2.find()) {
                    q2 = Double.parseDouble(m2.group(1));
                }

                if (q1 > q2) {
                    return -1;
                } else if (q1 == q2) {
                    return 0;
                } else {
                    return 1;
                }
            }
        });

        // Remove q-values
        final List<String> result = new ArrayList<String>(unsortedTypes.size());

        for (final String string : unsortedTypes) {
            result.add(string.split(";")[0].trim()); //$NON-NLS-1$
        }

        return result;
    }

}
