/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.timer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.elasticsearch.common.collect.MapBackedSet;
import org.elasticsearch.common.collect.ReusableIterator;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.thread.ThreadRenamingRunnable;
import org.elasticsearch.common.timer.Timeout;
import org.elasticsearch.common.timer.Timer;
import org.elasticsearch.common.timer.TimerTask;
import org.elasticsearch.common.util.concurrent.ConcurrentIdentityHashMap;

public class HashedWheelTimer
implements Timer {
    private final ESLogger logger;
    private static final AtomicInteger id = new AtomicInteger();
    private final Worker worker = new Worker();
    final Thread workerThread;
    final AtomicBoolean shutdown = new AtomicBoolean();
    private final long roundDuration;
    final long tickDuration;
    final Set<HashedWheelTimeout>[] wheel;
    final ReusableIterator<HashedWheelTimeout>[] iterators;
    final int mask;
    final ReadWriteLock lock = new ReentrantReadWriteLock();
    volatile int wheelCursor;

    public HashedWheelTimer(ESLogger logger, ThreadFactory threadFactory, long tickDuration, TimeUnit unit) {
        this(logger, threadFactory, tickDuration, unit, 512);
    }

    public HashedWheelTimer(ESLogger logger, ThreadFactory threadFactory, long tickDuration, TimeUnit unit, int ticksPerWheel) {
        this.logger = logger;
        if (threadFactory == null) {
            throw new NullPointerException("threadFactory");
        }
        if (unit == null) {
            throw new NullPointerException("unit");
        }
        if (tickDuration <= 0L) {
            throw new IllegalArgumentException("tickDuration must be greater than 0: " + tickDuration);
        }
        if (ticksPerWheel <= 0) {
            throw new IllegalArgumentException("ticksPerWheel must be greater than 0: " + ticksPerWheel);
        }
        this.wheel = HashedWheelTimer.createWheel(ticksPerWheel);
        this.iterators = HashedWheelTimer.createIterators(this.wheel);
        this.mask = this.wheel.length - 1;
        this.tickDuration = tickDuration = unit.toMillis(tickDuration);
        if (tickDuration == Long.MAX_VALUE || tickDuration >= Long.MAX_VALUE / (long)this.wheel.length) {
            throw new IllegalArgumentException("tickDuration is too long: " + tickDuration + ' ' + (Object)((Object)unit));
        }
        this.roundDuration = tickDuration * (long)this.wheel.length;
        this.workerThread = threadFactory.newThread(new ThreadRenamingRunnable(this.worker, "Hashed wheel timer #" + id.incrementAndGet()));
    }

    private static Set<HashedWheelTimeout>[] createWheel(int ticksPerWheel) {
        if (ticksPerWheel <= 0) {
            throw new IllegalArgumentException("ticksPerWheel must be greater than 0: " + ticksPerWheel);
        }
        if (ticksPerWheel > 0x40000000) {
            throw new IllegalArgumentException("ticksPerWheel may not be greater than 2^30: " + ticksPerWheel);
        }
        ticksPerWheel = HashedWheelTimer.normalizeTicksPerWheel(ticksPerWheel);
        Set[] wheel = new Set[ticksPerWheel];
        for (int i = 0; i < wheel.length; ++i) {
            wheel[i] = new MapBackedSet(new ConcurrentIdentityHashMap(16, 0.95f, 4));
        }
        return wheel;
    }

    private static ReusableIterator<HashedWheelTimeout>[] createIterators(Set<HashedWheelTimeout>[] wheel) {
        ReusableIterator[] iterators = new ReusableIterator[wheel.length];
        for (int i = 0; i < wheel.length; ++i) {
            iterators[i] = (ReusableIterator)wheel[i].iterator();
        }
        return iterators;
    }

    private static int normalizeTicksPerWheel(int ticksPerWheel) {
        int normalizedTicksPerWheel;
        for (normalizedTicksPerWheel = 1; normalizedTicksPerWheel < ticksPerWheel; normalizedTicksPerWheel <<= 1) {
        }
        return normalizedTicksPerWheel;
    }

    public synchronized void start() {
        if (this.shutdown.get()) {
            throw new IllegalStateException("cannot be started once stopped");
        }
        if (!this.workerThread.isAlive()) {
            this.workerThread.start();
        }
    }

    @Override
    public synchronized Set<Timeout> stop() {
        if (!this.shutdown.compareAndSet(false, true)) {
            return Collections.emptySet();
        }
        boolean interrupted = false;
        while (this.workerThread.isAlive()) {
            this.workerThread.interrupt();
            try {
                this.workerThread.join(100L);
            }
            catch (InterruptedException e) {
                interrupted = true;
            }
        }
        if (interrupted) {
            Thread.currentThread().interrupt();
        }
        HashSet<HashedWheelTimeout> unprocessedTimeouts = new HashSet<HashedWheelTimeout>();
        for (Set<HashedWheelTimeout> bucket : this.wheel) {
            unprocessedTimeouts.addAll(bucket);
            bucket.clear();
        }
        return Collections.unmodifiableSet(unprocessedTimeouts);
    }

    @Override
    public Timeout newTimeout(TimerTask task, long delay, TimeUnit unit) {
        long currentTime = System.currentTimeMillis();
        if (task == null) {
            throw new NullPointerException("task");
        }
        if (unit == null) {
            throw new NullPointerException("unit");
        }
        if ((delay = unit.toMillis(delay)) < this.tickDuration) {
            delay = this.tickDuration;
        }
        if (!this.workerThread.isAlive()) {
            this.start();
        }
        HashedWheelTimeout timeout = new HashedWheelTimeout(task, currentTime + delay);
        this.scheduleTimeout(timeout, delay);
        return timeout;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void scheduleTimeout(HashedWheelTimeout timeout, long delay) {
        if (delay < this.tickDuration) {
            delay = this.tickDuration;
        }
        long lastRoundDelay = delay % this.roundDuration;
        long lastTickDelay = delay % this.tickDuration;
        long relativeIndex = lastRoundDelay / this.tickDuration + (long)(lastTickDelay != 0L ? 1 : 0);
        long remainingRounds = delay / this.roundDuration - (long)(delay % this.roundDuration == 0L ? 1 : 0);
        this.lock.readLock().lock();
        try {
            int stopIndex;
            timeout.stopIndex = stopIndex = (int)((long)this.wheelCursor + relativeIndex & (long)this.mask);
            timeout.remainingRounds = remainingRounds;
            this.wheel[stopIndex].add(timeout);
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private final class HashedWheelTimeout
    implements Timeout {
        private final TimerTask task;
        final long deadline;
        volatile int stopIndex;
        volatile long remainingRounds;
        private volatile boolean cancelled;

        HashedWheelTimeout(TimerTask task, long deadline) {
            this.task = task;
            this.deadline = deadline;
        }

        @Override
        public Timer getTimer() {
            return HashedWheelTimer.this;
        }

        @Override
        public TimerTask getTask() {
            return this.task;
        }

        @Override
        public void cancel() {
            if (this.isExpired()) {
                return;
            }
            this.cancelled = true;
            HashedWheelTimer.this.wheel[this.stopIndex].remove(this);
        }

        @Override
        public boolean isCancelled() {
            return this.cancelled;
        }

        @Override
        public boolean isExpired() {
            return this.cancelled || System.currentTimeMillis() > this.deadline;
        }

        public void expire() {
            if (this.cancelled) {
                return;
            }
            try {
                this.task.run(this);
            }
            catch (Throwable t) {
                HashedWheelTimer.this.logger.warn("An exception was thrown by " + TimerTask.class.getSimpleName() + ".", t, new Object[0]);
            }
        }

        public String toString() {
            long currentTime = System.currentTimeMillis();
            long remaining = this.deadline - currentTime;
            StringBuilder buf = new StringBuilder(192);
            buf.append(this.getClass().getSimpleName());
            buf.append('(');
            buf.append("deadline: ");
            if (remaining > 0L) {
                buf.append(remaining);
                buf.append(" ms later, ");
            } else if (remaining < 0L) {
                buf.append(-remaining);
                buf.append(" ms ago, ");
            } else {
                buf.append("now, ");
            }
            if (this.isCancelled()) {
                buf.append(", cancelled");
            }
            return buf.append(')').toString();
        }
    }

    private final class Worker
    implements Runnable {
        private long startTime;
        private long tick;

        Worker() {
        }

        @Override
        public void run() {
            ArrayList<HashedWheelTimeout> expiredTimeouts = new ArrayList<HashedWheelTimeout>();
            this.startTime = System.currentTimeMillis();
            this.tick = 1L;
            while (!HashedWheelTimer.this.shutdown.get()) {
                long deadline = this.waitForNextTick();
                if (deadline <= 0L) continue;
                this.fetchExpiredTimeouts(expiredTimeouts, deadline);
                this.notifyExpiredTimeouts(expiredTimeouts);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void fetchExpiredTimeouts(List<HashedWheelTimeout> expiredTimeouts, long deadline) {
            HashedWheelTimer.this.lock.writeLock().lock();
            try {
                int newWheelCursor = HashedWheelTimer.this.wheelCursor = HashedWheelTimer.this.wheelCursor + 1 & HashedWheelTimer.this.mask;
                ReusableIterator<HashedWheelTimeout> i = HashedWheelTimer.this.iterators[newWheelCursor];
                this.fetchExpiredTimeouts(expiredTimeouts, i, deadline);
            }
            finally {
                HashedWheelTimer.this.lock.writeLock().unlock();
            }
        }

        private void fetchExpiredTimeouts(List<HashedWheelTimeout> expiredTimeouts, ReusableIterator<HashedWheelTimeout> i, long deadline) {
            ArrayList<HashedWheelTimeout> slipped = null;
            i.rewind();
            while (i.hasNext()) {
                HashedWheelTimeout timeout = (HashedWheelTimeout)i.next();
                if (timeout.remainingRounds <= 0L) {
                    i.remove();
                    if (timeout.deadline <= deadline) {
                        expiredTimeouts.add(timeout);
                        continue;
                    }
                    if (slipped == null) {
                        slipped = new ArrayList<HashedWheelTimeout>();
                    }
                    slipped.add(timeout);
                    continue;
                }
                --timeout.remainingRounds;
            }
            if (slipped != null) {
                for (HashedWheelTimeout timeout : slipped) {
                    HashedWheelTimer.this.scheduleTimeout(timeout, timeout.deadline - deadline);
                }
            }
        }

        private void notifyExpiredTimeouts(List<HashedWheelTimeout> expiredTimeouts) {
            for (int i = expiredTimeouts.size() - 1; i >= 0; --i) {
                expiredTimeouts.get(i).expire();
            }
            expiredTimeouts.clear();
        }

        private long waitForNextTick() {
            long currentTime;
            long sleepTime;
            long deadline = this.startTime + HashedWheelTimer.this.tickDuration * this.tick;
            while ((sleepTime = HashedWheelTimer.this.tickDuration * this.tick - ((currentTime = System.currentTimeMillis()) - this.startTime)) > 0L) {
                try {
                    Thread.sleep(sleepTime);
                }
                catch (InterruptedException e) {
                    if (!HashedWheelTimer.this.shutdown.get()) continue;
                    return -1L;
                }
            }
            ++this.tick;
            return deadline;
        }
    }
}

