/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.memory;

import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.index.service.IndexService;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.service.IndexShard;
import org.elasticsearch.index.shard.service.InternalIndexShard;
import org.elasticsearch.indices.IndicesLifecycle;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.monitor.jvm.JvmInfo;

public class IndexingMemoryBufferController
extends AbstractComponent {
    private final ByteSizeValue indexingBuffer;
    private final ByteSizeValue minShardIndexBufferSize;
    private final IndicesService indicesService;
    private final Listener listener = new Listener();

    @Inject
    public IndexingMemoryBufferController(Settings settings, IndicesService indicesService) {
        super(settings);
        ByteSizeValue indexingBuffer;
        this.indicesService = indicesService;
        String indexingBufferSetting = this.componentSettings.get("index_buffer_size", "10%");
        if (indexingBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(indexingBufferSetting.substring(0, indexingBufferSetting.length() - 1));
            indexingBuffer = new ByteSizeValue((long)((double)JvmInfo.jvmInfo().mem().heapMax().bytes() * (percent / 100.0)));
            ByteSizeValue minIndexingBuffer = this.componentSettings.getAsBytesSize("min_index_buffer_size", new ByteSizeValue(48L, ByteSizeUnit.MB));
            ByteSizeValue maxIndexingBuffer = this.componentSettings.getAsBytesSize("max_index_buffer_size", null);
            if (indexingBuffer.bytes() < minIndexingBuffer.bytes()) {
                indexingBuffer = minIndexingBuffer;
            }
            if (maxIndexingBuffer != null && indexingBuffer.bytes() > maxIndexingBuffer.bytes()) {
                indexingBuffer = maxIndexingBuffer;
            }
        } else {
            indexingBuffer = ByteSizeValue.parseBytesSizeValue(indexingBufferSetting, null);
        }
        this.indexingBuffer = indexingBuffer;
        this.minShardIndexBufferSize = this.componentSettings.getAsBytesSize("min_shard_index_buffer_size", new ByteSizeValue(4L, ByteSizeUnit.MB));
        this.logger.debug("using index_buffer_size [{}], with min_shard_index_buffer_size [{}]", this.indexingBuffer, this.minShardIndexBufferSize);
        indicesService.indicesLifecycle().addListener(this.listener);
    }

    private class Listener
    extends IndicesLifecycle.Listener {
        private Listener() {
        }

        @Override
        public void afterIndexShardCreated(IndexShard indexShard) {
            this.calcAndSetShardIndexingBuffer("created_shard[" + indexShard.shardId().index().name() + "][" + indexShard.shardId().id() + "]");
        }

        @Override
        public void afterIndexShardClosed(ShardId shardId, boolean delete) {
            this.calcAndSetShardIndexingBuffer("removed_shard[" + shardId.index().name() + "][" + shardId.id() + "]");
        }

        private void calcAndSetShardIndexingBuffer(String reason) {
            int shardsCount = this.countShards();
            if (shardsCount == 0) {
                return;
            }
            ByteSizeValue shardIndexingBufferSize = this.calcShardIndexingBuffer(shardsCount);
            if (shardIndexingBufferSize == null) {
                return;
            }
            if (shardIndexingBufferSize.bytes() < IndexingMemoryBufferController.this.minShardIndexBufferSize.bytes()) {
                shardIndexingBufferSize = IndexingMemoryBufferController.this.minShardIndexBufferSize;
            }
            IndexingMemoryBufferController.this.logger.debug("recalculating shard indexing buffer (reason={}), total is [{}] with [{}] shards, each shard set to [{}]", reason, IndexingMemoryBufferController.this.indexingBuffer, shardsCount, shardIndexingBufferSize);
            for (IndexService indexService : IndexingMemoryBufferController.this.indicesService) {
                for (IndexShard indexShard : indexService) {
                    ((InternalIndexShard)indexShard).engine().updateIndexingBufferSize(shardIndexingBufferSize);
                }
            }
        }

        private ByteSizeValue calcShardIndexingBuffer(int shardsCount) {
            return new ByteSizeValue(IndexingMemoryBufferController.this.indexingBuffer.bytes() / (long)shardsCount);
        }

        private int countShards() {
            int shardsCount = 0;
            for (IndexService indexService : IndexingMemoryBufferController.this.indicesService) {
                for (IndexShard indexShard : indexService) {
                    ++shardsCount;
                }
            }
            return shardsCount;
        }
    }
}

