/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.discovery.zen.fd;

import java.io.IOException;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import org.elasticsearch.ElasticSearchIllegalStateException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Streamable;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.discovery.zen.DiscoveryNodesProvider;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.BaseTransportRequestHandler;
import org.elasticsearch.transport.BaseTransportResponseHandler;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportConnectionListener;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportService;

public class MasterFaultDetection
extends AbstractComponent {
    private final ThreadPool threadPool;
    private final TransportService transportService;
    private final DiscoveryNodesProvider nodesProvider;
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList();
    private final boolean connectOnNetworkDisconnect;
    private final TimeValue pingInterval;
    private final TimeValue pingRetryTimeout;
    private final int pingRetryCount;
    private final boolean registerConnectionListener;
    private final FDConnectionListener connectionListener;
    private volatile MasterPinger masterPinger;
    private final Object masterNodeMutex = new Object();
    private volatile DiscoveryNode masterNode;
    private volatile int retryCount;
    private final AtomicBoolean notifiedMasterFailure = new AtomicBoolean();

    public MasterFaultDetection(Settings settings, ThreadPool threadPool, TransportService transportService, DiscoveryNodesProvider nodesProvider) {
        super(settings);
        this.threadPool = threadPool;
        this.transportService = transportService;
        this.nodesProvider = nodesProvider;
        this.connectOnNetworkDisconnect = this.componentSettings.getAsBoolean("connect_on_network_disconnect", true);
        this.pingInterval = this.componentSettings.getAsTime("ping_interval", TimeValue.timeValueSeconds(1L));
        this.pingRetryTimeout = this.componentSettings.getAsTime("ping_timeout", TimeValue.timeValueSeconds(30L));
        this.pingRetryCount = this.componentSettings.getAsInt("ping_retries", 3);
        this.registerConnectionListener = this.componentSettings.getAsBoolean("register_connection_listener", true);
        this.logger.debug("[master] uses ping_interval [{}], ping_timeout [{}], ping_retries [{}]", this.pingInterval, this.pingRetryTimeout, this.pingRetryCount);
        this.connectionListener = new FDConnectionListener();
        if (this.registerConnectionListener) {
            transportService.addConnectionListener(this.connectionListener);
        }
        transportService.registerHandler("discovery/zen/fd/masterPing", new MasterPingRequestHandler());
    }

    public DiscoveryNode masterNode() {
        return this.masterNode;
    }

    public void addListener(Listener listener) {
        this.listeners.add(listener);
    }

    public void removeListener(Listener listener) {
        this.listeners.remove(listener);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void restart(DiscoveryNode masterNode, String reason) {
        Object object = this.masterNodeMutex;
        synchronized (object) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("[master] restarting fault detection against master [{}], reason [{}]", masterNode, reason);
            }
            this.innerStop();
            this.innerStart(masterNode);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void start(DiscoveryNode masterNode, String reason) {
        Object object = this.masterNodeMutex;
        synchronized (object) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("[master] starting fault detection against master [{}], reason [{}]", masterNode, reason);
            }
            this.innerStart(masterNode);
        }
    }

    private void innerStart(DiscoveryNode masterNode) {
        this.masterNode = masterNode;
        this.retryCount = 0;
        this.notifiedMasterFailure.set(false);
        try {
            this.transportService.connectToNode(masterNode);
        }
        catch (Exception e) {
            this.notifyMasterFailure(masterNode, "failed to perform initial connect [" + e.getMessage() + "]");
            return;
        }
        if (this.masterPinger != null) {
            this.masterPinger.stop();
        }
        this.masterPinger = new MasterPinger();
        this.threadPool.schedule(this.pingInterval, "same", this.masterPinger);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop(String reason) {
        Object object = this.masterNodeMutex;
        synchronized (object) {
            if (this.masterNode != null && this.logger.isDebugEnabled()) {
                this.logger.debug("[master] stopping fault detection against master [{}], reason [{}]", this.masterNode, reason);
            }
            this.innerStop();
        }
    }

    private void innerStop() {
        this.retryCount = 0;
        if (this.masterPinger != null) {
            this.masterPinger.stop();
            this.masterPinger = null;
        }
        this.masterNode = null;
    }

    public void close() {
        this.stop("closing");
        this.listeners.clear();
        this.transportService.removeConnectionListener(this.connectionListener);
        this.transportService.removeHandler("discovery/zen/fd/masterPing");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleTransportDisconnect(DiscoveryNode node) {
        Object object = this.masterNodeMutex;
        synchronized (object) {
            if (!node.equals(this.masterNode)) {
                return;
            }
            if (this.connectOnNetworkDisconnect) {
                try {
                    this.transportService.connectToNode(node);
                    if (this.masterPinger != null) {
                        this.masterPinger.stop();
                    }
                    this.masterPinger = new MasterPinger();
                    this.threadPool.schedule(this.pingInterval, "same", this.masterPinger);
                }
                catch (Exception e) {
                    this.logger.trace("[master] [{}] transport disconnected (with verified connect)", this.masterNode);
                    this.notifyMasterFailure(this.masterNode, "transport disconnected (with verified connect)");
                }
            } else {
                this.logger.trace("[master] [{}] transport disconnected", node);
                this.notifyMasterFailure(node, "transport disconnected");
            }
        }
    }

    private void notifyDisconnectedFromMaster() {
        this.threadPool.cached().execute(new Runnable(){

            @Override
            public void run() {
                for (Listener listener : MasterFaultDetection.this.listeners) {
                    listener.onDisconnectedFromMaster();
                }
            }
        });
    }

    private void notifyMasterFailure(final DiscoveryNode masterNode, final String reason) {
        if (this.notifiedMasterFailure.compareAndSet(false, true)) {
            this.threadPool.cached().execute(new Runnable(){

                @Override
                public void run() {
                    for (Listener listener : MasterFaultDetection.this.listeners) {
                        listener.onMasterFailure(masterNode, reason);
                    }
                }
            });
            this.stop("master failure, " + reason);
        }
    }

    private static class MasterPingResponseResponse
    implements Streamable {
        private boolean connectedToMaster;

        private MasterPingResponseResponse() {
        }

        private MasterPingResponseResponse(boolean connectedToMaster) {
            this.connectedToMaster = connectedToMaster;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            this.connectedToMaster = in.readBoolean();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeBoolean(this.connectedToMaster);
        }
    }

    private static class MasterPingRequest
    implements Streamable {
        private String nodeId;
        private String masterNodeId;

        private MasterPingRequest() {
        }

        private MasterPingRequest(String nodeId, String masterNodeId) {
            this.nodeId = nodeId;
            this.masterNodeId = masterNodeId;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            this.nodeId = in.readUTF();
            this.masterNodeId = in.readUTF();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeUTF(this.nodeId);
            out.writeUTF(this.masterNodeId);
        }
    }

    private class MasterPingRequestHandler
    extends BaseTransportRequestHandler<MasterPingRequest> {
        public static final String ACTION = "discovery/zen/fd/masterPing";

        private MasterPingRequestHandler() {
        }

        @Override
        public MasterPingRequest newInstance() {
            return new MasterPingRequest();
        }

        @Override
        public void messageReceived(MasterPingRequest request, TransportChannel channel) throws Exception {
            DiscoveryNodes nodes = MasterFaultDetection.this.nodesProvider.nodes();
            if (!request.masterNodeId.equals(nodes.localNodeId())) {
                throw new ElasticSearchIllegalStateException("Got ping as master with id [" + request.masterNodeId + "], but not master and no id");
            }
            if (!nodes.localNodeMaster()) {
                throw new NoLongerMasterException();
            }
            channel.sendResponse(new MasterPingResponseResponse(nodes.nodeExists(request.nodeId)));
        }

        @Override
        public String executor() {
            return "same";
        }
    }

    private static class NoLongerMasterException
    extends ElasticSearchIllegalStateException {
        private NoLongerMasterException() {
        }

        @Override
        public Throwable fillInStackTrace() {
            return null;
        }
    }

    private class MasterPinger
    implements Runnable {
        private volatile boolean running = true;

        private MasterPinger() {
        }

        public void stop() {
            this.running = false;
        }

        @Override
        public void run() {
            if (!this.running) {
                return;
            }
            final DiscoveryNode masterToPing = MasterFaultDetection.this.masterNode;
            if (masterToPing == null) {
                MasterFaultDetection.this.threadPool.schedule(MasterFaultDetection.this.pingInterval, "same", this);
                return;
            }
            MasterFaultDetection.this.transportService.sendRequest(masterToPing, "discovery/zen/fd/masterPing", new MasterPingRequest(MasterFaultDetection.this.nodesProvider.nodes().localNode().id(), masterToPing.id()), TransportRequestOptions.options().withHighType().withTimeout(MasterFaultDetection.this.pingRetryTimeout), new BaseTransportResponseHandler<MasterPingResponseResponse>(){

                @Override
                public MasterPingResponseResponse newInstance() {
                    return new MasterPingResponseResponse();
                }

                @Override
                public void handleResponse(MasterPingResponseResponse response) {
                    if (!MasterPinger.this.running) {
                        return;
                    }
                    MasterFaultDetection.this.retryCount = 0;
                    if (masterToPing.equals(MasterFaultDetection.this.masterNode())) {
                        if (!response.connectedToMaster) {
                            MasterFaultDetection.this.logger.trace("[master] [{}] does not have us registered with it...", masterToPing);
                            MasterFaultDetection.this.notifyDisconnectedFromMaster();
                        }
                        MasterFaultDetection.this.threadPool.schedule(MasterFaultDetection.this.pingInterval, "same", MasterPinger.this);
                    }
                }

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void handleException(TransportException exp) {
                    if (!MasterPinger.this.running) {
                        return;
                    }
                    if (exp instanceof ConnectTransportException) {
                        return;
                    }
                    Object object = MasterFaultDetection.this.masterNodeMutex;
                    synchronized (object) {
                        if (masterToPing.equals(MasterFaultDetection.this.masterNode())) {
                            if (exp.getCause() instanceof NoLongerMasterException) {
                                MasterFaultDetection.this.logger.debug("[master] pinging a master {} that is no longer a master", MasterFaultDetection.this.masterNode, MasterFaultDetection.this.pingRetryCount, MasterFaultDetection.this.pingRetryTimeout);
                                MasterFaultDetection.this.notifyMasterFailure(masterToPing, "no longer master");
                            }
                            int retryCount = ++MasterFaultDetection.this.retryCount;
                            MasterFaultDetection.this.logger.trace("[master] failed to ping [{}], retry [{}] out of [{}]", exp, MasterFaultDetection.this.masterNode, retryCount, MasterFaultDetection.this.pingRetryCount);
                            if (retryCount >= MasterFaultDetection.this.pingRetryCount) {
                                MasterFaultDetection.this.logger.debug("[master] failed to ping [{}], tried [{}] times, each with maximum [{}] timeout", MasterFaultDetection.this.masterNode, MasterFaultDetection.this.pingRetryCount, MasterFaultDetection.this.pingRetryTimeout);
                                MasterFaultDetection.this.notifyMasterFailure(masterToPing, "failed to ping, tried [" + MasterFaultDetection.this.pingRetryCount + "] times, each with  maximum [" + MasterFaultDetection.this.pingRetryTimeout + "] timeout");
                            } else {
                                MasterFaultDetection.this.transportService.sendRequest(masterToPing, "discovery/zen/fd/masterPing", new MasterPingRequest(MasterFaultDetection.this.nodesProvider.nodes().localNode().id(), masterToPing.id()), TransportRequestOptions.options().withHighType().withTimeout(MasterFaultDetection.this.pingRetryTimeout), this);
                            }
                        }
                    }
                }

                @Override
                public String executor() {
                    return "same";
                }
            });
        }
    }

    private class FDConnectionListener
    implements TransportConnectionListener {
        private FDConnectionListener() {
        }

        @Override
        public void onNodeConnected(DiscoveryNode node) {
        }

        @Override
        public void onNodeDisconnected(DiscoveryNode node) {
            MasterFaultDetection.this.handleTransportDisconnect(node);
        }
    }

    public static interface Listener {
        public void onMasterFailure(DiscoveryNode var1, String var2);

        public void onDisconnectedFromMaster();
    }
}

