/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.memory;

import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import org.elasticsearch.ElasticSearchException;
import org.elasticsearch.common.collect.Maps;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.engine.EngineClosedException;
import org.elasticsearch.index.engine.FlushNotAllowedEngineException;
import org.elasticsearch.index.service.IndexService;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.service.IndexShard;
import org.elasticsearch.index.shard.service.InternalIndexShard;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.IndicesLifecycle;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.threadpool.ThreadPool;

public class IndexingMemoryBufferController
extends AbstractLifecycleComponent<IndexingMemoryBufferController> {
    private final ThreadPool threadPool;
    private final IndicesService indicesService;
    private final ByteSizeValue indexingBuffer;
    private final ByteSizeValue minShardIndexBufferSize;
    private final ByteSizeValue maxShardIndexBufferSize;
    private final TimeValue inactiveTime;
    private final TimeValue interval;
    private final Listener listener = new Listener();
    private final Map<ShardId, ShardIndexingStatus> shardsIndicesStatus = Maps.newHashMap();
    private volatile ScheduledFuture scheduler;
    private final Object mutex = new Object();

    @Inject
    public IndexingMemoryBufferController(Settings settings, ThreadPool threadPool, IndicesService indicesService) {
        super(settings);
        ByteSizeValue indexingBuffer;
        this.threadPool = threadPool;
        this.indicesService = indicesService;
        String indexingBufferSetting = this.componentSettings.get("index_buffer_size", "10%");
        if (indexingBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(indexingBufferSetting.substring(0, indexingBufferSetting.length() - 1));
            indexingBuffer = new ByteSizeValue((long)((double)JvmInfo.jvmInfo().mem().heapMax().bytes() * (percent / 100.0)));
            ByteSizeValue minIndexingBuffer = this.componentSettings.getAsBytesSize("min_index_buffer_size", new ByteSizeValue(48L, ByteSizeUnit.MB));
            ByteSizeValue maxIndexingBuffer = this.componentSettings.getAsBytesSize("max_index_buffer_size", null);
            if (indexingBuffer.bytes() < minIndexingBuffer.bytes()) {
                indexingBuffer = minIndexingBuffer;
            }
            if (maxIndexingBuffer != null && indexingBuffer.bytes() > maxIndexingBuffer.bytes()) {
                indexingBuffer = maxIndexingBuffer;
            }
        } else {
            indexingBuffer = ByteSizeValue.parseBytesSizeValue(indexingBufferSetting, null);
        }
        this.indexingBuffer = indexingBuffer;
        this.minShardIndexBufferSize = this.componentSettings.getAsBytesSize("min_shard_index_buffer_size", new ByteSizeValue(4L, ByteSizeUnit.MB));
        this.maxShardIndexBufferSize = this.componentSettings.getAsBytesSize("max_shard_index_buffer_size", new ByteSizeValue(512L, ByteSizeUnit.MB));
        this.inactiveTime = this.componentSettings.getAsTime("shard_inactive_time", TimeValue.timeValueMinutes(30L));
        this.interval = this.componentSettings.getAsTime("interval", TimeValue.timeValueSeconds(30L));
        this.logger.debug("using index_buffer_size [{}], with min_shard_index_buffer_size [{}], max_shard_index_buffer_size [{}], shard_inactive_time [{}]", this.indexingBuffer, this.minShardIndexBufferSize, this.maxShardIndexBufferSize, this.inactiveTime);
    }

    @Override
    protected void doStart() throws ElasticSearchException {
        this.indicesService.indicesLifecycle().addListener(this.listener);
        this.scheduler = this.threadPool.scheduleWithFixedDelay(new ShardsIndicesStatusChecker(), this.interval);
    }

    @Override
    protected void doStop() throws ElasticSearchException {
        this.indicesService.indicesLifecycle().removeListener(this.listener);
        if (this.scheduler != null) {
            this.scheduler.cancel(false);
            this.scheduler = null;
        }
    }

    @Override
    protected void doClose() throws ElasticSearchException {
    }

    private void calcAndSetShardIndexingBuffer(String reason) {
        int shardsCount = this.countShards();
        if (shardsCount == 0) {
            return;
        }
        ByteSizeValue shardIndexingBufferSize = this.calcShardIndexingBuffer(shardsCount);
        if (shardIndexingBufferSize == null) {
            return;
        }
        if (shardIndexingBufferSize.bytes() < this.minShardIndexBufferSize.bytes()) {
            shardIndexingBufferSize = this.minShardIndexBufferSize;
        }
        if (shardIndexingBufferSize.bytes() > this.maxShardIndexBufferSize.bytes()) {
            shardIndexingBufferSize = this.maxShardIndexBufferSize;
        }
        this.logger.debug("recalculating shard indexing buffer (reason={}), total is [{}] with [{}] active shards, each shard set to [{}]", reason, this.indexingBuffer, shardsCount, shardIndexingBufferSize);
        for (IndexService indexService : this.indicesService) {
            for (IndexShard indexShard : indexService) {
                ShardIndexingStatus status = this.shardsIndicesStatus.get(indexShard.shardId());
                if (status != null && status.inactive) continue;
                try {
                    ((InternalIndexShard)indexShard).engine().updateIndexingBufferSize(shardIndexingBufferSize);
                }
                catch (EngineClosedException e) {
                }
                catch (FlushNotAllowedEngineException e) {
                }
                catch (Exception e) {
                    this.logger.warn("failed to set shard [{}][{}] index buffer to [{}]", indexShard.shardId().index().name(), indexShard.shardId().id(), shardIndexingBufferSize);
                }
            }
        }
    }

    private ByteSizeValue calcShardIndexingBuffer(int shardsCount) {
        return new ByteSizeValue(this.indexingBuffer.bytes() / (long)shardsCount);
    }

    private int countShards() {
        int shardsCount = 0;
        for (IndexService indexService : this.indicesService) {
            for (IndexShard indexShard : indexService) {
                ShardIndexingStatus status = this.shardsIndicesStatus.get(indexShard.shardId());
                if (status != null && status.inactive) continue;
                ++shardsCount;
            }
        }
        return shardsCount;
    }

    static class ShardIndexingStatus {
        long translogId = -1L;
        int translogNumberOfOperations = -1;
        boolean inactive = false;
        long time = -1L;

        ShardIndexingStatus() {
        }
    }

    class Listener
    extends IndicesLifecycle.Listener {
        Listener() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void afterIndexShardCreated(IndexShard indexShard) {
            Object object = IndexingMemoryBufferController.this.mutex;
            synchronized (object) {
                IndexingMemoryBufferController.this.calcAndSetShardIndexingBuffer("created_shard[" + indexShard.shardId().index().name() + "][" + indexShard.shardId().id() + "]");
                IndexingMemoryBufferController.this.shardsIndicesStatus.put(indexShard.shardId(), new ShardIndexingStatus());
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void afterIndexShardClosed(ShardId shardId, boolean delete) {
            Object object = IndexingMemoryBufferController.this.mutex;
            synchronized (object) {
                IndexingMemoryBufferController.this.calcAndSetShardIndexingBuffer("removed_shard[" + shardId.index().name() + "][" + shardId.id() + "]");
                IndexingMemoryBufferController.this.shardsIndicesStatus.remove(shardId);
            }
        }
    }

    class ShardsIndicesStatusChecker
    implements Runnable {
        ShardsIndicesStatusChecker() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Object object = IndexingMemoryBufferController.this.mutex;
            synchronized (object) {
                boolean activeInactiveStatusChanges = false;
                for (IndexService indexService : IndexingMemoryBufferController.this.indicesService) {
                    for (IndexShard indexShard : indexService) {
                        long time = IndexingMemoryBufferController.this.threadPool.estimatedTimeInMillis();
                        Translog translog = ((InternalIndexShard)indexShard).translog();
                        ShardIndexingStatus status = (ShardIndexingStatus)IndexingMemoryBufferController.this.shardsIndicesStatus.get(indexShard.shardId());
                        if (status == null) continue;
                        if (status.translogId == translog.currentId() && translog.estimatedNumberOfOperations() == 0) {
                            if (status.time == -1L) {
                                status.time = time;
                            }
                            if (!status.inactive && time - status.time > IndexingMemoryBufferController.this.inactiveTime.millis()) {
                                try {
                                    ((InternalIndexShard)indexShard).engine().updateIndexingBufferSize(Engine.INACTIVE_SHARD_INDEXING_BUFFER);
                                }
                                catch (EngineClosedException e) {
                                    continue;
                                }
                                catch (FlushNotAllowedEngineException e) {
                                    continue;
                                }
                                status.inactive = true;
                                activeInactiveStatusChanges = true;
                                IndexingMemoryBufferController.this.logger.debug("marking shard [{}][{}] as inactive (inactive_time[{}]), setting size to [{}]", indexShard.shardId().index().name(), indexShard.shardId().id(), IndexingMemoryBufferController.this.inactiveTime, Engine.INACTIVE_SHARD_INDEXING_BUFFER);
                            }
                        } else {
                            if (status.inactive) {
                                status.inactive = false;
                                activeInactiveStatusChanges = true;
                                IndexingMemoryBufferController.this.logger.debug("marking shard [{}][{}] as active", indexShard.shardId().index().name(), indexShard.shardId().id());
                            }
                            status.time = -1L;
                        }
                        status.translogId = translog.currentId();
                        status.translogNumberOfOperations = translog.estimatedNumberOfOperations();
                    }
                }
                if (activeInactiveStatusChanges) {
                    IndexingMemoryBufferController.this.calcAndSetShardIndexingBuffer("shards became active/inactive");
                }
            }
        }
    }
}

