package play.modules.facebook;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.restfb.DefaultFacebookClient;
import com.restfb.FacebookClient;
import java.util.HashMap;
import java.util.Map;
import play.exceptions.UnexpectedException;
import play.libs.WS;
import play.libs.WS.HttpResponse;
import play.mvc.Http.Cookie;
import play.mvc.Http.Request;
import play.Play;

/**
 * FbGraph provides simple access to the Facebook Graph API.
 *
 * @author Eric Jacob
 */
public class FbGraph {

    public final static String FB_GRAPH_URL = "https://graph.facebook.com/";
    private static String appId;
    private static String appSecret;

    public static void init() {
        if (!Play.configuration.containsKey("fbg.appId")) {
            throw new UnexpectedException("Module FbGraph requires that you specify fbg.appId in your application.conf");
        }
        if (!Play.configuration.containsKey("fbg.appSecret")) {
            throw new UnexpectedException("Module FbGraph requires that you specify fbg.appSecret in your application.conf");
        }
        appId = Play.configuration.getProperty("fbg.appId");
        appSecret = Play.configuration.getProperty("fbg.appSecret");
    }

    /**
     * Returns the Id associated to this application.
     *
     * @return  the application Id
     */
    public static String getAppId() {
        return appId;
    }

    /**
     * Returns the Secret associated to this application.
     *
     * @return  the application Secret
     */
    public static String getAppSecret() {
        return appSecret;
    }

    private static Map<String, String> parseStr(String str) {
        String[] pairs = str.split("&");
        Map<String, String> map = new HashMap<String, String>();
        for (String pair : pairs) {
            String[] kv = pair.split("=");
            map.put(kv[0], kv[1]);
        }
        return map;
    }

    /**
     * Returns the data stored in the Facebook session associated with the user.
     *
     * @return  the data stored in the Facebook cookie
     *          or null if no Facebook session associated with the user
     */
    public static Map getFacebookCookie() {
        Map<String, String> fbData = null;
        Cookie cookie = Request.current().cookies.get("fbs_" + getAppId());
        if (cookie != null) {
            fbData = parseStr(cookie.value);
        }
        return fbData;
    }

    /**
     * Returns the access token associated with this application.
     *
     * @return  the application access token
     *          or null if no Facebook session associated with the user
     */
    public static String getAccessToken() {
        String accessToken = null;
        if (getFacebookCookie() != null) {
            accessToken = (String) getFacebookCookie().get("access_token");
        }
        return accessToken;
    }

    /**
     * Executes a GET or POST request to the Graph API.
     *
     * @param   path the URL path
     * @param   method the HTTP method (optional, default "GET")
     * @param   params the parameters for the query (optional)
     * @return  the HTTP response
     */
    private static HttpResponse makeRequest(String path, String method, Map<String, String> params) {
        StringBuilder url = new StringBuilder();
        url.append(FB_GRAPH_URL);
        if (path.startsWith("/")) {
            path = path.substring(1);
        }
        url.append(path);
        if (method != null && method.toUpperCase().equals("POST")) {
            return WS.url(url.toString()).setParameters(params).post();
        } else {
            url.append("?");
            StringBuilder queryStr = new StringBuilder();
            for (Map.Entry<String, String> param : params.entrySet()) {
                if (queryStr.length() > 0) {
                    queryStr.append("&");
                }
                queryStr.append(WS.encode(param.getKey()));
                queryStr.append("=");
                queryStr.append(WS.encode(param.getValue()));
            }
            url.append(queryStr.toString());
            return WS.url(url.toString()).get();
        }
    }

    /**
     * Performs an authorized request to the Graph API.
     *
     * @param   path the URL path
     * @param   method the HTTP method (optional, default "GET")
     * @param   params the parameters for the query (optional)
     * @return  the HTTP response
     */
    private static HttpResponse oauthRequest(String path, String method, Map<String, String> params) {
        if (params == null) {
            params = new HashMap<String, String>();
        }
        if (!params.containsKey("access_token")) {
            params.put("access_token", getAccessToken());
        }
        return makeRequest(path, method, params);
    }

    /**
     * Executes an API call to the Graph API.
     *
     * @param   path the URL path, e.g. "me/friends"
     * @param   method the HTTP method (optional, default "GET")
     * @param   params the parameters for the query (optional)
     * @return  the response object
     *          or null if no Facebook session associated with the user
     * @throws  FbGraphException
     */
    public static JsonElement api(String path, String method, Map<String, String> params)
            throws FbGraphException {
        JsonElement json = null;
        if (getFacebookCookie() != null) {
            HttpResponse resp = oauthRequest(path, method, params);
            if (resp == null) {
                throw new UnexpectedException("Module FbGraph got an unexpected response from facebook");
            }
            if (resp.getStatus() != 200) {
                throw new FbGraphException("HttpResponse", Integer.toString(resp.getStatus()), resp.getString());
            }
            json = resp.getJson();
            if (json.isJsonObject()) {
                JsonObject jsonObject = (JsonObject) json;
                if (jsonObject.get("error") != null) {
                    throw new FbGraphException(jsonObject);
                }
            }
        }
        return json;
    }

    /**
     * Executes an API call to the Graph API.
     *
     * @param   path the URL path, e.g. "me/friends"
     * @param   method the HTTP method (optional, default "GET")
     * @return  the response object
     *          or null if no Facebook session associated with the user
     * @throws  FbGraphException
     */
    public static JsonElement api(String path, String method)
            throws FbGraphException {
        return api(path, method, null);
    }

    /**
     * Executes an API call to the Graph API.
     *
     * @param   path the URL path, e.g. "me/friends"
     * @param   params the parameters for the query (optional)
     * @return  the response object
     *          or null if no Facebook session associated with the user
     * @throws  FbGraphException
     */
    public static JsonElement api(String path, Map<String, String> params)
            throws FbGraphException {
        return api(path, "GET", params);
    }

    /**
     * Executes an API call to the Graph API.
     *
     * @param   path the URL path, e.g. "me/friends"
     * @return  the response object
     *          or null if no Facebook session associated with the user
     * @throws  FbGraphException
     */
    public static JsonElement api(String path) throws FbGraphException {
        return api(path, "GET", null);
    }

    /**
     * Fetches a single object to the Graph API.
     *
     * @param   object the ID of the object, e.g. "me"
     * @param   params the parameters for the query (optional)
     * @return  the response object
     *          or null if no Facebook session associated with the user
     * @throws  FbGraphException
     */
    public static JsonObject getObject(String object, Map<String, String> params)
            throws FbGraphException {
        return api(object, params).getAsJsonObject();
    }

    /**
     * Fetches a single Graph API object.
     *
     * @param   object the ID of the object, e.g. "me"
     * @return  the response object
     *          or null if no Facebook session associated with the user
     * @throws  FbGraphException
     */
    public static JsonObject getObject(String object) throws FbGraphException {
        return api(object).getAsJsonObject();
    }

    /**
     * Fetches a Graph API connection.
     *
     * @param   connection the ID/CONNECTION_TYPE string, e.g. "me/friends"
     * @param   params the parameters for the query (optional)
     * @return  the response object
     *          or null if no Facebook session associated with the user
     * @throws  FbGraphException
     */
    public static JsonArray getConnection(String connection, Map<String, String> params)
            throws FbGraphException {
        return api(connection, params).getAsJsonObject().get("data").getAsJsonArray();
    }

    /**
     * Fetches a Graph API connection.
     *
     * @param   connection the ID/CONNECTION_TYPE string, e.g. "me/friends"
     * @return  the response object
     *          or null if no Facebook session associated with the user
     * @throws  FbGraphException
     */
    public static JsonArray getConnection(String connection)
            throws FbGraphException {
        return getConnection(connection, null);
    }

    /**
     * Returns a picture URL.
     *
     * @param   id the ID of the picture
     * @return  the URL of the picture
     */
    public static String getPicture(String id) {
        return FB_GRAPH_URL + id + "/picture?access_token=" + WS.encode(getAccessToken());
    }

    /**
     * Returns a picture URL.
     *
     * @param   id the ID of the picture
     * @param   type the size of the picture
     * @return  the URL of the picture
     */
    public static String getPicture(String id, String type) {
        return getPicture(id) + "&type=" + type;
    }

    /**
     * Performs a Graph API publish operation.
     *
     * @param   path the URL path
     * @param   params the parameters of the post
     * @return  the published Facebook graph object
     * @throws  FbGraphException
     */
    public static JsonObject publish(String path, Map<String, String> params)
            throws FbGraphException {
        return api(path, "POST", params).getAsJsonObject();
    }

    /**
     * Performs a Graph API delete operation.
     *
     * @param   object the ID of the object
     * @return  true if successful, false otherwise
     * @throws  FbGraphException
     */
    public static Boolean delete(String object)
            throws FbGraphException {
        return api(object, "POST", Parameter.with("method", "delete").parameters()).getAsBoolean();
    }

    /**
     * Returns a RestFB Facebook client.
     *
     * @return  the Facebook client
     */
    public static FacebookClient getFacebookClient() {
        return new DefaultFacebookClient(getAccessToken());
    }
}
