/*
 * Decompiled with CFR 0.152.
 */
package com.google.appengine.tools.remoteapi;

import com.google.appengine.repackaged.com.google.io.protocol.ProtocolMessage;
import com.google.appengine.tools.remoteapi.RemoteApiException;
import com.google.appengine.tools.remoteapi.RemoteApiOptions;
import com.google.appengine.tools.remoteapi.RemoteRpc;
import com.google.appengine.tools.remoteapi.TransactionBuilder;
import com.google.apphosting.api.ApiBasePb;
import com.google.apphosting.api.DatastorePb;
import com.google.storage.onestore.v3.OnestoreEntity;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;

class RemoteDatastore {
    static final String DATASTORE_SERVICE = "datastore_v3";
    private static final String REMOTE_API_SERVICE = "remote_datastore";
    private static final Logger logger = Logger.getLogger(RemoteDatastore.class.getName());
    private final RemoteRpc remoteRpc;
    private final RemoteApiOptions options;
    private final String remoteAppId;
    private final Map<Long, QueryState> idToCursor = new ConcurrentHashMap<Long, QueryState>();
    private final AtomicLong nextCursorId = new AtomicLong(1L);
    private final Map<Long, TransactionBuilder> idToTransaction = new ConcurrentHashMap<Long, TransactionBuilder>();
    private final AtomicLong nextTransactionId = new AtomicLong(1L);

    RemoteDatastore(RemoteRpc remoteRpc, RemoteApiOptions options) {
        this.remoteRpc = remoteRpc;
        this.options = options;
        this.remoteAppId = remoteRpc.getClient().getAppId();
    }

    byte[] handleDatastoreCall(String methodName, byte[] request) {
        if (methodName.equals("RunQuery")) {
            return this.handleRunQuery(request);
        }
        if (methodName.equals("Next")) {
            return this.handleNext(request);
        }
        if (methodName.equals("BeginTransaction")) {
            return this.handleBeginTransaction();
        }
        if (methodName.equals("Commit")) {
            return this.handleCommit(request);
        }
        if (methodName.equals("Rollback")) {
            return this.handleRollback(request);
        }
        if (methodName.equals("Get")) {
            return this.handleGet(request);
        }
        if (methodName.equals("Put")) {
            return this.handlePut(request);
        }
        if (methodName.equals("Delete")) {
            return this.handleDelete(request);
        }
        return this.remoteRpc.call(DATASTORE_SERVICE, methodName, "", request);
    }

    private byte[] handleRunQuery(byte[] request) {
        return this.runQuery(request, this.nextCursorId.getAndIncrement());
    }

    private byte[] runQuery(byte[] request, long localCursorId) {
        DatastorePb.Query query = new DatastorePb.Query();
        query.mergeFrom(request);
        if (RemoteDatastore.rewriteQueryAppIds(query, this.remoteAppId)) {
            request = query.toByteArray();
        }
        query.setCompile(true);
        if (!query.hasCount()) {
            query.setCount(this.options.getDatastoreQueryFetchSize());
        }
        TransactionBuilder tx = null;
        if (query.hasTransaction()) {
            tx = this.getTransactionBuilder("RunQuery", query.getTransaction());
            query.clearTransaction();
        }
        byte[] resultBytes = this.remoteRpc.call(DATASTORE_SERVICE, "RunQuery", "", query.toByteArray());
        DatastorePb.QueryResult result = new DatastorePb.QueryResult();
        result.mergeFrom(resultBytes);
        if (result.isMoreResults() && result.hasCompiledCursor()) {
            this.idToCursor.put(localCursorId, new QueryState(request, result.getCompiledCursor()));
        } else {
            this.idToCursor.put(localCursorId, QueryState.NO_MORE_RESULTS);
        }
        if (tx != null) {
            for (OnestoreEntity.EntityProto entity : result.results()) {
                tx.addEntityToCache(entity);
            }
        }
        result.getMutableCursor().setCursor(localCursorId);
        return result.toByteArray();
    }

    static boolean rewriteQueryAppIds(DatastorePb.Query query, String remoteAppId) {
        boolean reserialize = false;
        if (!query.getApp().equals(remoteAppId)) {
            reserialize = true;
            query.setApp(remoteAppId);
        }
        for (DatastorePb.Query.Filter filter : query.filters()) {
            for (OnestoreEntity.Property prop : filter.propertys()) {
                OnestoreEntity.PropertyValue.ReferenceValue ref;
                OnestoreEntity.PropertyValue propValue = prop.getMutableValue();
                if (!propValue.hasReferenceValue() || (ref = propValue.getMutableReferenceValue()).getApp().equals(remoteAppId)) continue;
                reserialize = true;
                ref.setApp(remoteAppId);
            }
        }
        return reserialize;
    }

    private byte[] handleNext(byte[] request) {
        DatastorePb.NextRequest nextRequest = new DatastorePb.NextRequest();
        nextRequest.mergeFrom(request);
        long cursorId = nextRequest.getCursor().getCursor();
        QueryState queryState = this.idToCursor.get(cursorId);
        if (queryState == null) {
            throw new RemoteApiException("local cursor not found", DATASTORE_SERVICE, "Next", null);
        }
        if (!queryState.hasMoreResults()) {
            DatastorePb.QueryResult result = new DatastorePb.QueryResult();
            result.setMoreResults(false);
            return result.toByteArray();
        }
        return this.runQuery(queryState.makeNextQuery(nextRequest).toByteArray(), cursorId);
    }

    private byte[] handleBeginTransaction() {
        long txId = this.nextTransactionId.getAndIncrement();
        this.idToTransaction.put(txId, new TransactionBuilder());
        DatastorePb.Transaction tx = new DatastorePb.Transaction();
        tx.setHandle(txId);
        tx.setApp(this.remoteAppId);
        return tx.toByteArray();
    }

    private byte[] handleCommit(byte[] requestBytes) {
        DatastorePb.Transaction request = new DatastorePb.Transaction();
        request.mergeFrom(requestBytes);
        request.setApp(this.remoteAppId);
        TransactionBuilder tx = this.removeTransactionBuilder("Commit", request);
        this.remoteRpc.call(REMOTE_API_SERVICE, "Transaction", "", tx.makeCommitRequest().toByteArray());
        return new DatastorePb.CommitResponse().toByteArray();
    }

    private byte[] handleRollback(byte[] requestBytes) {
        DatastorePb.Transaction request = new DatastorePb.Transaction();
        request.mergeFrom(requestBytes);
        request.setApp(this.remoteAppId);
        this.removeTransactionBuilder("Rollback", request);
        return new ApiBasePb.VoidProto().toByteArray();
    }

    private byte[] handleGet(byte[] requestBytes) {
        DatastorePb.GetRequest request = new DatastorePb.GetRequest();
        request.mergeFrom(requestBytes);
        if (RemoteDatastore.rewriteReferenceAppIds(request.mutableKeys(), this.remoteAppId)) {
            requestBytes = request.toByteArray();
        }
        if (request.hasTransaction()) {
            return this.handleGetForTransaction(request);
        }
        return this.remoteRpc.call(DATASTORE_SERVICE, "Get", "", requestBytes);
    }

    private byte[] handlePut(byte[] requestBytes) {
        DatastorePb.PutRequest request = new DatastorePb.PutRequest();
        request.mergeFrom(requestBytes);
        boolean reserialize = RemoteDatastore.rewritePutAppIds(request, this.remoteAppId);
        if (request.hasTransaction()) {
            return this.handlePutForTransaction(request);
        }
        if (reserialize) {
            requestBytes = request.toByteArray();
        }
        String suffix = "";
        if (logger.isLoggable(Level.FINE)) {
            suffix = RemoteDatastore.describePutRequestForLog(request);
        }
        return this.remoteRpc.call(DATASTORE_SERVICE, "Put", suffix, requestBytes);
    }

    static boolean rewritePutAppIds(DatastorePb.PutRequest request, String remoteAppId) {
        boolean reserialize = false;
        for (OnestoreEntity.EntityProto entity : request.mutableEntitys()) {
            if (!entity.getMutableKey().getApp().equals(remoteAppId)) {
                reserialize = true;
                entity.getMutableKey().setApp(remoteAppId);
            }
            for (OnestoreEntity.Property prop : entity.mutablePropertys()) {
                OnestoreEntity.PropertyValue.ReferenceValue ref;
                if (!prop.hasValue() || !prop.getMutableValue().hasReferenceValue() || !(ref = prop.getMutableValue().getReferenceValue()).hasApp() || ref.getApp().equals(remoteAppId)) continue;
                reserialize = true;
                ref.setApp(remoteAppId);
            }
        }
        return reserialize;
    }

    private byte[] handleDelete(byte[] requestBytes) {
        DatastorePb.DeleteRequest request = new DatastorePb.DeleteRequest();
        request.mergeFrom(requestBytes);
        if (RemoteDatastore.rewriteReferenceAppIds(request.mutableKeys(), this.remoteAppId)) {
            requestBytes = request.toByteArray();
        }
        if (request.hasTransaction()) {
            return this.handleDeleteForTransaction(request);
        }
        return this.remoteRpc.call(DATASTORE_SERVICE, "Delete", "", requestBytes);
    }

    static boolean rewriteReferenceAppIds(List<OnestoreEntity.Reference> references, String remoteAppId) {
        boolean reserialize = false;
        for (OnestoreEntity.Reference ref : references) {
            if (ref.getApp().equals(remoteAppId)) continue;
            reserialize = true;
            ref.setApp(remoteAppId);
        }
        return reserialize;
    }

    byte[] handleGetForTransaction(DatastorePb.GetRequest request) {
        TransactionBuilder tx = this.getTransactionBuilder("Get", request.getTransaction());
        DatastorePb.GetRequest subRequest = new DatastorePb.GetRequest();
        for (OnestoreEntity.Reference key : request.keys()) {
            if (tx.isCachedEntity(key)) continue;
            subRequest.addKey(key);
        }
        if (subRequest.keySize() > 0) {
            byte[] subResponseBytes = this.remoteRpc.call(DATASTORE_SERVICE, "Get", "", subRequest.toByteArray());
            DatastorePb.GetResponse subResponse = new DatastorePb.GetResponse();
            subResponse.mergeFrom(subResponseBytes);
            Iterator keys = request.keys().iterator();
            for (DatastorePb.GetResponse.Entity entityPb : subResponse.entitys()) {
                OnestoreEntity.Reference key = (OnestoreEntity.Reference)keys.next();
                if (entityPb.hasEntity()) {
                    tx.addEntityToCache(entityPb.getEntity());
                    continue;
                }
                tx.addEntityAbsenceToCache(key);
            }
        }
        DatastorePb.GetResponse mergedResponse = new DatastorePb.GetResponse();
        for (OnestoreEntity.Reference key : request.keys()) {
            OnestoreEntity.EntityProto entity = tx.getCachedEntity(key);
            if (entity == null) {
                mergedResponse.addEntity();
                continue;
            }
            mergedResponse.addEntity().setEntity(entity);
        }
        return mergedResponse.toByteArray();
    }

    byte[] handlePutForTransaction(DatastorePb.PutRequest request) {
        TransactionBuilder tx = this.getTransactionBuilder("Put", request.getTransaction());
        ArrayList<OnestoreEntity.EntityProto> entitiesWithoutIds = new ArrayList<OnestoreEntity.EntityProto>();
        for (OnestoreEntity.EntityProto entity : request.entitys()) {
            if (!this.requiresId(entity)) continue;
            entitiesWithoutIds.add(entity);
        }
        if (entitiesWithoutIds.size() > 0) {
            DatastorePb.PutRequest subRequest = new DatastorePb.PutRequest();
            for (OnestoreEntity.EntityProto entity : entitiesWithoutIds) {
                OnestoreEntity.EntityProto subEntity = subRequest.addEntity();
                subEntity.getKey().mergeFrom(entity.getKey());
                subEntity.getEntityGroup();
            }
            byte[] subResponseBytes = this.remoteRpc.call(REMOTE_API_SERVICE, "GetIDs", "", subRequest.toByteArray());
            DatastorePb.PutResponse subResponse = new DatastorePb.PutResponse();
            subResponse.mergeFrom(subResponseBytes);
            Iterator it = entitiesWithoutIds.iterator();
            for (OnestoreEntity.Reference newKey : subResponse.keys()) {
                OnestoreEntity.EntityProto entity = (OnestoreEntity.EntityProto)it.next();
                entity.getKey().copyFrom((ProtocolMessage)newKey);
                entity.getEntityGroup().addElement().copyFrom((ProtocolMessage)newKey.getPath().getElement(0));
            }
        }
        DatastorePb.PutResponse response = new DatastorePb.PutResponse();
        for (OnestoreEntity.EntityProto entityProto : request.entitys()) {
            tx.putEntityOnCommit(entityProto);
            response.addKey().copyFrom((ProtocolMessage)entityProto.getKey());
        }
        return response.toByteArray();
    }

    byte[] handleDeleteForTransaction(DatastorePb.DeleteRequest request) {
        TransactionBuilder tx = this.getTransactionBuilder("Delete", request.getTransaction());
        for (OnestoreEntity.Reference key : request.keys()) {
            tx.deleteEntityOnCommit(key);
        }
        DatastorePb.DeleteResponse response = new DatastorePb.DeleteResponse();
        return response.toByteArray();
    }

    TransactionBuilder getTransactionBuilder(String methodName, DatastorePb.Transaction tx) {
        TransactionBuilder result = this.idToTransaction.get(tx.getHandle());
        if (result == null) {
            throw new RemoteApiException("transaction not found", DATASTORE_SERVICE, methodName, null);
        }
        return result;
    }

    TransactionBuilder removeTransactionBuilder(String methodName, DatastorePb.Transaction tx) {
        TransactionBuilder result = this.idToTransaction.remove(tx.getHandle());
        if (result == null) {
            throw new RemoteApiException("transaction not found", DATASTORE_SERVICE, methodName, null);
        }
        return result;
    }

    private boolean requiresId(OnestoreEntity.EntityProto entity) {
        OnestoreEntity.Path path = entity.getKey().getPath();
        OnestoreEntity.Path.Element lastElement = (OnestoreEntity.Path.Element)path.elements().get(path.elementSize() - 1);
        return lastElement.getId() == 0L && !lastElement.hasName();
    }

    private static String describePutRequestForLog(DatastorePb.PutRequest putRequest) {
        int count = putRequest.entitySize();
        if (count <= 0) {
            return "()";
        }
        OnestoreEntity.Reference keyProto = putRequest.getEntity(0).getKey();
        if (count == 1) {
            return "(" + RemoteDatastore.describeKeyForLog(keyProto) + ")";
        }
        return "(" + RemoteDatastore.describeKeyForLog(keyProto) + ", ...)";
    }

    private static String describeKeyForLog(OnestoreEntity.Reference keyProto) {
        StringBuilder pathString = new StringBuilder();
        OnestoreEntity.Path path = keyProto.getPath();
        for (OnestoreEntity.Path.Element element : path.elements()) {
            if (pathString.length() > 0) {
                pathString.append(",");
            }
            pathString.append(element.getType() + "/");
            if (element.hasId()) {
                pathString.append(element.getId());
                continue;
            }
            pathString.append(element.getName());
        }
        return "[" + pathString + "]";
    }

    private static class QueryState {
        private static final QueryState NO_MORE_RESULTS = new QueryState(null, null);
        private final byte[] query;
        private final DatastorePb.CompiledCursor cursor;

        QueryState(byte[] query, DatastorePb.CompiledCursor cursor) {
            this.query = query;
            this.cursor = cursor;
        }

        boolean hasMoreResults() {
            return this.query != null;
        }

        private DatastorePb.Query makeNextQuery(DatastorePb.NextRequest nextRequest) {
            DatastorePb.Query result = new DatastorePb.Query();
            result.mergeFrom(this.query);
            result.setOffset(0);
            result.setCompiledCursor(this.cursor);
            result.setCompile(true);
            if (nextRequest.hasCount()) {
                result.setCount(nextRequest.getCount());
            } else {
                result.clearCount();
            }
            return result;
        }
    }
}

