/**
 * Copyright (c) 2011 Piero Sartini
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package play.modules.jersey;

import com.sun.jersey.api.container.ContainerFactory;
import play.Logger;
import play.Play;
import play.PlayPlugin;
import play.classloading.ApplicationClasses;
import play.modules.jersey.provider.DefaultJacksonProvider;
import play.modules.jersey.provider.DefaultJacksonXMLProvider;

import javax.ws.rs.Path;
import javax.ws.rs.ext.Provider;
import java.lang.annotation.Annotation;
import java.lang.reflect.Modifier;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author Piero Sartini
 */
public class JerseyPlugin extends PlayPlugin {

    public PlayContainer container;
    final Set<Class<?>> resourceClasses = new HashSet<Class<?>>();
    public String path;
    private boolean started;
    public final static String LOG_PREP = "Jersey plugin: ";

    @Override
    public void onConfigurationRead() {
        log("Configuration read");
        path = Play.configuration.getProperty("jersey.path");
        if (path == null)
            path = "/rest";
    }

    @Override
    public void onApplicationStart() {
        try {
            if (!started) {
                log("Starting Jersey");
                createContainer();
                log("Jersey started");
                started = true;
            } else if (Play.mode != Play.Mode.PROD) {
                createContainer();
            }
        } catch (Throwable t) {
            t.printStackTrace();
            throw new RuntimeException(t);
        }
    }

    private void createContainer() {
        resourceClasses.clear();

        // classes
        List<ApplicationClasses.ApplicationClass> classes = Play.classes.all();

        // default providers
        log("Default Providers: " + DefaultJacksonProvider.class.getName() + ", " + DefaultJacksonXMLProvider.class.getName());
        resourceClasses.add(DefaultJacksonProvider.class);
        resourceClasses.add(DefaultJacksonXMLProvider.class);

        for (ApplicationClasses.ApplicationClass ac : classes) {
            if (isRessource(ac.javaClass)) {
                log("Discovered Ressource class: " + ac.javaClass);
                resourceClasses.add(ac.javaClass);
            }
        }
        container = ContainerFactory.createContainer(PlayContainer.class, resourceClasses);
    }

    private boolean isRessource(Class<?> javaClass) {
        // ignore interfaces and abstract classes
        if (javaClass.isInterface()
                || Modifier.isAbstract(javaClass.getModifiers()))
            return false;

        // check for @Path or @Provider annotation
        if (hasAnnotation(javaClass, Path.class)
                || hasAnnotation(javaClass, Provider.class))
            return true;

        return false;
    }

    private boolean hasAnnotation(Class<?> type, Class<? extends Annotation> annotation) {
        // null cannot have annotations
        if (type == null)
            return false;

        // class annotation
        if (type.isAnnotationPresent(annotation))
            return true;

        // annotation on interface
        for (Class<?> interfaceType : type.getInterfaces()) {
            if (hasAnnotation(interfaceType, annotation))
                return true;
        }

        // annotation on superclass
        return hasAnnotation(type.getSuperclass(), annotation);

    }

    public static void log(String message, Object... params) {
        Logger.info(LOG_PREP + message, params);
    }
}
