/**
 * This file is part of LogiSima.
 *
 * LogiSima is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * LogiSima is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LogiSima. If not, see <http://www.gnu.org/licenses/>.
 */
package play.modules.yml;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.persistence.Embedded;
import javax.persistence.Id;
import javax.persistence.Lob;
import javax.persistence.ManyToMany;
import javax.persistence.OneToMany;
import javax.persistence.Temporal;

import org.hibernate.Hibernate;
import org.hibernate.proxy.HibernateProxy;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.DumperOptions.ScalarStyle;
import org.yaml.snakeyaml.Yaml;

import play.Logger;
import play.Play;
import play.db.jpa.GenericModel;
import play.db.jpa.JPABase;
import play.db.jpa.Model;
import play.modules.yml.models.YmlObject;

/**
 * Util class for logisima-yml module.
 * 
 * @author bsimard
 * 
 */
public class YmlExtractorUtil {

    private static final String TAB = " ";

    /**
     * Method that generate the YLM file.
     * 
     * @param output
     * @param filename
     * @param myHash
     * @throws IOException
     */
    public static void writeYml(String output, String filename) throws IOException {
        // we create the file
        File file = new File(output + "/" + filename + ".yml");
        FileOutputStream fop = new FileOutputStream(file);
        fop.write("# Generated by logisima-play-yml (http://github.com/sim51/logisima-play-yml).\n".getBytes());
        fop.write("# This module is a part of LogiSima (http://www.logisima.com).\n".getBytes());
        Iterator it = YmlExtractor.ymlObjects.entrySet().iterator();
        String tmp = "";
        while (it.hasNext()) {
            Entry object = (Entry) it.next();
            YmlObject ymlObject = (YmlObject) object.getValue();
            if (!ymlObject.isAlreadyWrite()) {
                tmp = writeObject2Yml(ymlObject, 0);
                fop.write(tmp.getBytes());
            }

        }
        fop.flush();
        fop.close();
    }

    /**
     * Recursive method to write object.
     * 
     * @param objectMap
     * @param object
     * @return
     */
    public static String writeObject2Yml(YmlObject object, int level) {
        level++;
        String ymlText = "";
        if (!object.isAlreadyWrite()) {
            // we mark the object as write !
            object.setAlreadyWrite(Boolean.TRUE);
            YmlExtractor.ymlObjects.put(object.getId(), object);

            if (object.getChildren().size() != 0) {
                for (int i = 0; i < object.getChildren().size(); i++) {
                    ymlText += writeObject2Yml(YmlExtractor.ymlObjects.get(object.getChildren().get(i)), level);
                }
            }
            ymlText += object.getYmlValue();
            // a hack for embedded object and !!class
            ymlText = ymlText.replaceAll("!!models.*", "");
            // a hack to convert braces into unicode due to groovy template
            ymlText = ymlText.replaceAll("[{]", "\\\\u007B");
            ymlText = ymlText.replaceAll("[}]", "\\\\u007D");
        }
        return ymlText;
    }

    /**
     * Method to convert an object to YmlObject.
     * 
     * @param jpaSupport
     * @return
     * @throws IllegalArgumentException
     * @throws IllegalAccessException
     * @throws ParseException
     * @throws NoSuchMethodException
     * @throws SecurityException
     * @throws InvocationTargetException
     */
    public static YmlObject object2YmlObject(JPABase jpaBase) throws IllegalArgumentException, IllegalAccessException,
            ParseException, SecurityException, NoSuchMethodException, InvocationTargetException {
        // Init YAML
        DumperOptions options = new DumperOptions();
        options.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK);
        options.setDefaultScalarStyle(ScalarStyle.DOUBLE_QUOTED);
        options.setCanonical(false);
        options.setAllowUnicode(true);
        Yaml yaml = new Yaml(options);

        // Initialization of YmlObject
        YmlObject ymlObject = new YmlObject();
        ymlObject.setId(getObjectId(jpaBase));

        // String value for the object
        String stringObject = "\n" + getObjectClassName(jpaBase) + "(" + getObjectId(jpaBase) + "):\n";
        Logger.info("Generate YML for class id :" + getObjectId(jpaBase) + " (" + jpaBase.getClass().getFields().length
                + " fields)");

        // if class is a javassist class
        if (jpaBase.getClass().getCanonicalName().contains("_$$_")) {
            Hibernate.initialize(jpaBase);
            HibernateProxy proxy = (HibernateProxy) jpaBase;
            jpaBase = (JPABase) proxy.getHibernateLazyInitializer().getImplementation();
        }

        for (java.lang.reflect.Field field : jpaBase.getClass().getFields()) {
            Map<String, Object> data = new HashMap<String, Object>();

            String name = field.getName();

            if (!Modifier.isStatic(field.getModifiers()) && !name.equals("id") && !name.equals("willBeSaved")
                    && !isFieldHasMappedByInAnnotation(field)) {

                Boolean valueIsSet = Boolean.FALSE;
                Logger.debug("Generated field " + name);

                if (field.get(jpaBase) != null) {

                    // if field is a List
                    if (List.class.isInstance(field.get(jpaBase))) {
                        Logger.debug("Field " + name + " type is List");
                        List myList = (List) field.get(jpaBase);
                        if (!myList.isEmpty() && myList.size() > 0) {
                            String[] tmpValues = new String[myList.size()];
                            for (int i = 0; i < myList.size(); i++) {
                                // if myObj is an entity, we add it to children
                                if (GenericModel.class.isInstance(myList.get(i))) {
                                    tmpValues[i] = getObjectId(myList.get(i));
                                    ymlObject.getChildren().add(getObjectId(myList.get(i)));
                                }
                                else {
                                    tmpValues[i] = myList.get(i).toString();
                                }
                            }
                            data.put(name, tmpValues);
                        }
                        valueIsSet = Boolean.TRUE;
                    }

                    // if field is a Map
                    if (Map.class.isInstance(field.get(jpaBase))) {
                        Logger.debug("Field " + name + " type is Map");
                        Map myMap = (Map) field.get(jpaBase);
                        if (myMap != null && myMap.size() > 0) {
                            String[] tmpValues = new String[myMap.size()];
                            Iterator it = myMap.entrySet().iterator();
                            int i = 0;
                            while (it.hasNext()) {
                                Object myObj = it.next();
                                // if myObj is an entity, we add it to children
                                if (myObj != null && GenericModel.class.isInstance(myObj)) {
                                    tmpValues[i] = getObjectId(myObj);
                                    if (getObjectId(myObj) != null) {
                                        ymlObject.getChildren().add(getObjectId(myObj));
                                    }
                                }
                                else {
                                    tmpValues[i] = myObj.toString();
                                }
                                i++;
                            }
                            data.put(name, tmpValues);
                        }
                        valueIsSet = Boolean.TRUE;
                    }

                    // if field is a Set
                    if (Set.class.isInstance(field.get(jpaBase))) {
                        Logger.debug("Field " + name + " type is Set");
                        Set mySet = (Set) field.get(jpaBase);
                        if (mySet != null && mySet.size() > 0) {
                            String[] tmpValues = new String[mySet.size()];
                            Iterator it = mySet.iterator();
                            int i = 0;
                            while (it.hasNext()) {
                                Object myObj = it.next();
                                // if myObj is an entity, we add it to children
                                if (myObj != null && GenericModel.class.isInstance(myObj)) {
                                    tmpValues[i] = getObjectId(myObj);
                                    if (getObjectId(myObj) != null) {
                                        ymlObject.getChildren().add(getObjectId(myObj));
                                    }
                                }
                                else {
                                    tmpValues[i] = myObj.toString();
                                }
                                i++;
                            }
                            data.put(name, tmpValues);
                        }
                        valueIsSet = Boolean.TRUE;
                    }

                    // if Lob annotation, then bigtext
                    if (field.isAnnotationPresent(Lob.class)) {
                        Logger.debug("Field " + name + " type is a Lob");
                        if (field.get(jpaBase) != null) {
                            data.put(name, field.get(jpaBase).toString());
                        }
                        valueIsSet = Boolean.TRUE;
                    }

                    // if field is an object that extend Model
                    if (jpaBase != null && GenericModel.class.isInstance(field.get(jpaBase))) {
                        Logger.debug("Field " + name + " type is a Model");
                        ymlObject.getChildren().add(getObjectId(field.get(jpaBase)));
                        data.put(name, getObjectId(field.get(jpaBase)));
                        valueIsSet = Boolean.TRUE;
                    }

                    // if field is a date
                    if (Date.class.isInstance(field.get(jpaBase))) {
                        Logger.debug("Field " + name + " type is Date");
                        // In case of temporal JPA annotation
                        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        if (field.getAnnotation(Temporal.class) != null) {
                            Temporal temporal = field.getAnnotation(Temporal.class);
                            switch (temporal.value()) {
                                case DATE:
                                    sdf = new SimpleDateFormat("yyyy-MM-dd");
                                    break;
                                case TIME:
                                    sdf = new SimpleDateFormat("hh:mm:ss");
                                    break;
                                case TIMESTAMP:
                                    sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                                    break;
                            }
                        }

                        Date myDate = (Date) sdf.parse(field.get(jpaBase).toString());
                        SimpleDateFormat sd = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
                        data.put(name, sd.format(myDate));
                        valueIsSet = Boolean.TRUE;
                    }

                    // if field is a calendar
                    if (Calendar.class.isInstance(field.get(jpaBase))) {
                        Logger.debug("Field " + name + " type is Calendar");
                        Calendar cal = (Calendar) field.get(jpaBase);
                        Date date = cal.getTime();
                        SimpleDateFormat sd = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
                        data.put(name, sd.format(date));
                        valueIsSet = Boolean.TRUE;
                    }

                    // if field is an Embedded
                    if (field.isAnnotationPresent(Embedded.class)) {
                        Logger.debug("Field " + name + " is an embedded");
                        data.put(field.getName(), field.get(jpaBase));
                        valueIsSet = Boolean.TRUE;
                    }

                    // otherwise ...
                    if (!valueIsSet) {
                        Logger.debug("Field " + name + " type is Basic");
                        String tmpValue = "" + field.get(jpaBase);
                        data.put(name, tmpValue);
                        valueIsSet = Boolean.TRUE;
                    }

                    if (valueIsSet && !data.isEmpty()) {
                        // yml indentation
                        String value = yaml.dump(data).replaceAll("^", TAB);
                        // a little hack for scalar ... I have to find a
                        // better solution
                        value = value.replaceAll("- ", TAB + "- ");
                        // a little hack for tab String empty
                        stringObject += value;
                    }
                }
            }
        }
        ymlObject.setYmlValue(stringObject);

        return ymlObject;
    }

    /**
     * Method that return classname the object.
     * 
     * @param Object
     * @return real className
     */
    public static String getObjectClassName(Object object) {
        String classname = object.getClass().getName();
        if (classname.contains("_$$_")) {
            classname = classname.split("_")[0];
        }
        return classname;
    }

    /**
     * Method that return an id for the object.
     * 
     * @param Object
     * @return the id field value
     * @throws IllegalAccessException
     * @throws IllegalArgumentException
     */
    public static String getObjectId(Object object) throws IllegalArgumentException, IllegalAccessException {
        JPABase jpaBase = (JPABase) object;
        String objectId = null;
        // if the object extend from the play's model class
        if (jpaBase != null && jpaBase instanceof Model) {
            // we take the model id
            Model myModel = ((Model) jpaBase);
            objectId = getObjectClassName(object);
            objectId = objectId.replaceAll("[.]", "_");
            objectId += "_";
            objectId += myModel.getId();
        }
        // else we try to get value of the field with id annotation
        else {
            // we look up for the field with the id annotation
            Field fieldId = null;
            for (java.lang.reflect.Field field : jpaBase.getClass().getFields()) {
                if (field.getAnnotation(Id.class) != null) {
                    fieldId = field;
                }
            }
            if (fieldId != null) {
                objectId = getObjectClassName(object);
                objectId = objectId.replaceAll("[.]", "_");
                objectId += "_";
                objectId += fieldId.get(jpaBase).toString();
            }
        }

        return objectId;
    }

    /**
     * Method to get the DB dialect. Note: this method is a copy of play! framework code (but it's private ...)
     * 
     * @param driver
     * @return String
     */
    public static String getDefaultDialect(String driver) {
        if (driver != null && driver.equals("org.hsqldb.jdbcDriver")) {
            return "org.hibernate.dialect.HSQLDialect";
        }
        else if (driver != null && driver.equals("com.mysql.jdbc.Driver")) {
            return "play.db.jpa.MySQLDialect";
        }
        else {
            String dialect = Play.configuration.getProperty("jpa.dialect");
            if (dialect != null) {
                return dialect;
            }
            throw new UnsupportedOperationException("I do not know which hibernate dialect to use with " + driver
                    + ", use the property jpa.dialect in config file");
        }
    }

    public static Boolean isFieldHasMappedByInAnnotation(Field field) {
        Boolean bool = Boolean.FALSE;

        // try with ManyToMany
        if (field.isAnnotationPresent(ManyToMany.class)) {
            ManyToMany annotation = field.getAnnotation(ManyToMany.class);
            if (!annotation.mappedBy().isEmpty()) {
                bool = Boolean.TRUE;
            }
        }
        // try with OneToMany
        if (field.isAnnotationPresent(OneToMany.class)) {
            OneToMany annotation = field.getAnnotation(OneToMany.class);
            if (!annotation.mappedBy().isEmpty()) {
                bool = Boolean.TRUE;
            }
        }

        return bool;
    }

}
