/*
 * {{{ header & license
 * Copyright (c) 2004, 2005, 2008 Joshua Marinacci, Patrick Wright
 * Copyright (c) 2008 Patrick Wright
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * }}}
 */
package org.xhtmlrenderer.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.log4j.Logger;

/**
 * Utility class for using the java.util.logging package. Relies on the standard
 * configuration for logging, but gives easier access to the various logs
 * (plumbing.load, .init, .render)
 * 
 * @author empty
 */
public class XRLog {
	// MUST BE FIRST
	private static final List LOGGER_NAMES = new ArrayList(20);
	public static final class XRLoggerLog4JImpl implements XRLogger {
		@Override
		public void log(final String where, final Level level, final String msg) {
			final Logger log = Logger.getLogger(where);
			if (level.equals(Level.FINEST) || level.equals(Level.FINER)) {
				if (log.isTraceEnabled()) {
					log.trace(msg);
				}
			} else if (level.equals(Level.FINE)) {
				if (log.isDebugEnabled()) {
					log.debug(msg);
				}
			} else if (level.equals(Level.INFO)) {
				if (log.isInfoEnabled()) {
					log.info(msg);
				}
			} else if (level.equals(Level.WARNING)) {
				if (log.isEnabledFor(org.apache.log4j.Level.WARN)) {
					log.warn(msg);
				}
			} else if (level.equals(Level.SEVERE)) {
				if (log.isEnabledFor(org.apache.log4j.Level.ERROR)) {
					log.error(msg);
				}
			}
		}

		@Override
		public void log(final String where, final Level level,
				final String msg, final Throwable th) {
			final Logger log = Logger.getLogger(where);
			if (level.equals(Level.FINEST) || level.equals(Level.FINER)) {
				if (log.isTraceEnabled()) {
					log.trace(msg, th);
				}
			} else if (level.equals(Level.FINE)) {
				if (log.isDebugEnabled()) {
					log.debug(msg, th);
				}
			} else if (level.equals(Level.INFO)) {
				if (log.isInfoEnabled()) {
					log.info(msg, th);
				}
			} else if (level.equals(Level.WARNING)) {
				if (log.isEnabledFor(org.apache.log4j.Level.WARN)) {
					log.warn(msg, th);
				}
			} else if (level.equals(Level.SEVERE)) {
				if (log.isEnabledFor(org.apache.log4j.Level.ERROR)) {
					log.error(msg, th);
				}
			}
		}

		@Override
		public void setLevel(final String logger, final Level level) {
			final Logger log = Logger.getLogger(logger);
			if (level.equals(Level.FINEST) || level.equals(Level.FINER)) {
				log.setLevel(org.apache.log4j.Level.TRACE);
			} else if (level.equals(Level.FINE)) {
				log.setLevel(org.apache.log4j.Level.DEBUG);
			} else if (level.equals(Level.INFO)) {
				log.setLevel(org.apache.log4j.Level.INFO);
			} else if (level.equals(Level.WARNING)) {
				log.setLevel(org.apache.log4j.Level.WARN);
			} else if (level.equals(Level.SEVERE)) {
				log.setLevel(org.apache.log4j.Level.ERROR);
			}
		}
	}

	public final static String CASCADE = XRLog
			.registerLoggerByName("org.xhtmlrenderer.cascade");
	public final static String CONFIG = XRLog
			.registerLoggerByName("org.xhtmlrenderer.config");
	public final static String CSS_PARSE = XRLog
			.registerLoggerByName("org.xhtmlrenderer.css-parse");
	public final static String EXCEPTION = XRLog
			.registerLoggerByName("org.xhtmlrenderer.exception");
	public final static String GENERAL = XRLog
			.registerLoggerByName("org.xhtmlrenderer.general");
	public final static String INIT = XRLog
			.registerLoggerByName("org.xhtmlrenderer.init");
	private static boolean initPending = true;
	public final static String JUNIT = XRLog
			.registerLoggerByName("org.xhtmlrenderer.junit");
	public final static String LAYOUT = XRLog
			.registerLoggerByName("org.xhtmlrenderer.layout");
	public final static String LOAD = XRLog
			.registerLoggerByName("org.xhtmlrenderer.load");
	private static final XRLogger loggerImpl = new XRLoggerLog4JImpl();

	public final static String MATCH = XRLog
			.registerLoggerByName("org.xhtmlrenderer.match");

	public final static String RENDER = XRLog
			.registerLoggerByName("org.xhtmlrenderer.render");;

	public final static String XML_ENTITIES = XRLog
			.registerLoggerByName("org.xhtmlrenderer.load.xml-entities");

	public static void cascade(final Level level, final String msg) {
		XRLog.log(XRLog.CASCADE, level, msg);
	}

	public static void cascade(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.CASCADE, level, msg, th);
	}

	public static void cascade(final String msg) {
		XRLog.cascade(Level.INFO, msg);
	}

	public static void cssParse(final Level level, final String msg) {
		XRLog.log(XRLog.CSS_PARSE, level, msg);
	}

	public static void cssParse(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.CSS_PARSE, level, msg, th);
	}

	public static void cssParse(final String msg) {
		XRLog.cssParse(Level.INFO, msg);
	}

	public static void exception(final String msg) {
		XRLog.exception(msg, null);
	}

	public static void exception(final String msg, final Throwable th) {
		XRLog.log(XRLog.EXCEPTION, Level.WARNING, msg, th);
	}

	public static void general(final Level level, final String msg) {
		XRLog.log(XRLog.GENERAL, level, msg);
	}

	public static void general(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.GENERAL, level, msg, th);
	}

	public static void general(final String msg) {
		XRLog.general(Level.INFO, msg);
	}

	public static XRLogger getLoggerImpl() {
		return XRLog.loggerImpl;
	}

	private static void init() {
		synchronized (XRLog.class) {
			if (!XRLog.initPending) {
				return;
			}
			// now change this immediately, in case something fails
			XRLog.initPending = false;

			XRLog.setLoggingEnabled(Configuration.isTrue(
					"xr.util-logging.loggingEnabled", true));

			if (XRLog.loggerImpl == null) {
				// loggerImpl = new JDKXRLogger();
			}
		}
	}

	public static void init(final Level level, final String msg) {
		XRLog.log(XRLog.INIT, level, msg);
	}

	public static void init(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.INIT, level, msg, th);
	}

	public static void init(final String msg) {
		XRLog.init(Level.INFO, msg);
	}

	/**
	 * Whether logging is on or off.
	 * 
	 * @return Returns true if logging is enabled, false if not. Corresponds to
	 *         configuration file property xr.util-logging.loggingEnabled, or to
	 *         value passed to setLoggingEnabled(bool).
	 */
	@Deprecated
	public static boolean isLoggingEnabled() {
		return true;
	}

	public static void junit(final Level level, final String msg) {
		XRLog.log(XRLog.JUNIT, level, msg);
	}

	public static void junit(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.JUNIT, level, msg, th);
	}

	public static void junit(final String msg) {
		XRLog.junit(Level.FINEST, msg);
	}

	public static void layout(final Level level, final String msg) {
		XRLog.log(XRLog.LAYOUT, level, msg);
	}

	public static void layout(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.LAYOUT, level, msg, th);
	}

	public static void layout(final String msg) {
		XRLog.layout(Level.INFO, msg);
	}

	/**
	 * Returns a list of all loggers that will be accessed by XRLog. Each entry
	 * is a String with a logger name, which can be used to retrieve the logger
	 * using the corresponding Logging API; example name might be
	 * "org.xhtmlrenderer.config"
	 * 
	 * @return List of loggers, never null.
	 */
	public static List listRegisteredLoggers() {
		// defensive copy
		return new ArrayList(XRLog.LOGGER_NAMES);
	}

	public static void load(final Level level, final String msg) {
		XRLog.log(XRLog.LOAD, level, msg);
	}

	public static void load(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.LOAD, level, msg, th);
	}

	public static void load(final String msg) {
		XRLog.load(Level.INFO, msg);
	}

	public static void log(final String where, final Level level,
			final String msg) {
		if (XRLog.initPending) {
			XRLog.init();
		}
		if (XRLog.isLoggingEnabled()) {
			XRLog.loggerImpl.log(where, level, msg);
		}
	}

	public static void log(final String where, final Level level,
			final String msg, final Throwable th) {
		if (XRLog.initPending) {
			XRLog.init();
		}
		if (XRLog.isLoggingEnabled()) {
			XRLog.loggerImpl.log(where, level, msg, th);
		}
	}

	public static void main(final String args[]) {
		try {
			XRLog.cascade("Cascade msg");
			XRLog.cascade(Level.WARNING, "Cascade msg");
			XRLog.exception("Exception msg");
			XRLog.exception("Exception msg", new Exception());
			XRLog.general("General msg");
			XRLog.general(Level.WARNING, "General msg");
			XRLog.init("Init msg");
			XRLog.init(Level.WARNING, "Init msg");
			XRLog.load("Load msg");
			XRLog.load(Level.WARNING, "Load msg");
			XRLog.match("Match msg");
			XRLog.match(Level.WARNING, "Match msg");
			XRLog.layout("Layout msg");
			XRLog.layout(Level.WARNING, "Layout msg");
			XRLog.render("Render msg");
			XRLog.render(Level.WARNING, "Render msg");
		} catch (final Exception ex) {
			ex.printStackTrace();
		}
	}

	public static void match(final Level level, final String msg) {
		XRLog.log(XRLog.MATCH, level, msg);
	}

	public static void match(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.MATCH, level, msg, th);
	}

	public static void match(final String msg) {
		XRLog.match(Level.INFO, msg);
	}

	private static String registerLoggerByName(final String loggerName) {
		XRLog.LOGGER_NAMES.add(loggerName);
		return loggerName;
	}

	public static void render(final Level level, final String msg) {
		XRLog.log(XRLog.RENDER, level, msg);
	}

	public static void render(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.RENDER, level, msg, th);
	}

	public static void render(final String msg) {
		XRLog.render(Level.INFO, msg);
	}

	public static void setLevel(final String log, final Level level) {
		if (XRLog.initPending) {
			XRLog.init();
		}
		XRLog.loggerImpl.setLevel(log, level);
	}

	@Deprecated
	public static void setLoggerImpl(final XRLogger loggerImpl) {
	}

	@Deprecated
	public static void setLoggingEnabled(final boolean loggingEnabled) {
	}

	public static void xmlEntities(final Level level, final String msg) {
		XRLog.log(XRLog.XML_ENTITIES, level, msg);
	}

	public static void xmlEntities(final Level level, final String msg,
			final Throwable th) {
		XRLog.log(XRLog.XML_ENTITIES, level, msg, th);
	}

	public static void xmlEntities(final String msg) {
		XRLog.xmlEntities(Level.INFO, msg);
	}
}// end class

/*
 * $Id: XRLog.java,v 1.19 2008/01/27 16:40:29 pdoubleya Exp $
 * 
 * $Log: XRLog.java,v $ Revision 1.19 2008/01/27 16:40:29 pdoubleya Issues 186
 * and 130: fix configuration so that logging setup does not override any
 * current settings for JDK logging classes. Disable logging by default.
 * 
 * Revision 1.18 2007/09/10 20:28:26 peterbrant Make underlying logging
 * implementation pluggable / Add log4j logging implementation (not currently
 * compiled with Ant to avoid additional compile time dependency)
 * 
 * Revision 1.17 2007/06/02 20:00:34 peterbrant Revert earlier change to default
 * CSS parse logging level / Use WARNING explicitly for CSS parse errors
 * 
 * Revision 1.16 2007/06/01 21:44:08 peterbrant CSS parsing errors should be
 * logged at WARNING, not INFO level
 * 
 * Revision 1.15 2006/08/17 17:32:25 joshy intial patch to fix the logging
 * config issues https://xhtmlrenderer.dev.java.net/issues/show_bug.cgi?id=130
 * 
 * Revision 1.14 2006/07/26 17:59:01 pdoubleya Use proper form for logging
 * exceptions.
 * 
 * Revision 1.13 2006/07/17 22:15:59 pdoubleya Added loggingEnabled switch to
 * XRLog and config file; default logging to off there and in Configuration. Fix
 * for Issue Tracker #123.
 * 
 * Revision 1.12 2005/07/13 22:49:15 joshy updates to get the jnlp to work
 * without being signed
 * 
 * Revision 1.11 2005/06/26 01:21:35 tobega Fixed possible infinite loop in
 * init()
 * 
 * Revision 1.10 2005/05/06 16:54:32 joshy forgot to add this level stuff
 * 
 * Issue number: Obtained from: Submitted by: Reviewed by:
 * 
 * Revision 1.9 2005/04/07 16:14:28 pdoubleya Updated to clarify relationship
 * between Configuration and XRLog on load; Configuration must load first, but
 * holds off on logging until XRLog is initialized. LogStartupConfig no longer
 * used.
 * 
 * Revision 1.8 2005/03/27 18:36:26 pdoubleya Added separate logging for entity
 * resolution.
 * 
 * Revision 1.7 2005/01/29 20:18:38 pdoubleya Clean/reformat code. Removed
 * commented blocks, checked copyright.
 * 
 * Revision 1.6 2005/01/29 12:18:15 pdoubleya Added cssParse logging.
 * 
 * Revision 1.5 2005/01/24 19:01:10 pdoubleya Mass checkin. Changed to use
 * references to CSSName, which now has a Singleton instance for each property,
 * everywhere property names were being used before. Removed commented code.
 * Cascaded and Calculated style now store properties in arrays rather than
 * maps, for optimization.
 * 
 * Revision 1.4 2005/01/24 14:33:07 pdoubleya Added junit logging hierarchy.
 * 
 * Revision 1.3 2004/10/23 14:06:57 pdoubleya Re-formatted using JavaStyle tool.
 * Cleaned imports to resolve wildcards except for common packages (java.io,
 * java.util, etc). Added CVS log comments at bottom.
 */

