package play.modules.vhost;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import play.Logger;
import play.Play;
import play.PlayPlugin;
import play.utils.Properties;

import com.den_4.inotify_java.Constants;
import com.den_4.inotify_java.EventQueueFull;
import com.den_4.inotify_java.Inotify;
import com.den_4.inotify_java.InotifyEvent;
import com.den_4.inotify_java.InotifyEventListener;
import com.mchange.v2.c3p0.ComboPooledDataSource;

public class VirtualHostsPlugin extends PlayPlugin
{
  private static VirtualHostsPlugin config          = null;

  private String                    configDir       = null;
  private Inotify                   iNotify         = null;
  private InotifyEventListener      iNotifyListener = null;
  private int                       iNotifyWatch;
  private Map<String, VirtualHost>  vHostMap        = new HashMap<String, VirtualHost>();
  private Map<String, VirtualHost>  fqdnMap         = new HashMap<String, VirtualHost>();

  static VirtualHost findHost(String domain)
  {
    return config != null ? config.fqdnMap.get(domain) : null;
  }

  static VirtualHost[] getAllHosts()
  {
    final VirtualHost[] empty = {};
    return config != null ? config.fqdnMap.values().toArray(empty) : empty;
  }

  public VirtualHostsPlugin()
  {
    super();
    if (config == null) config = this;
  }

  public VirtualHostsPlugin(String pConfigDir)
  {
    super();
    if (config == null) config = this;
    config.loadVirtualHostRegistrations(pConfigDir);
  }

  @Override
  public void onApplicationStart()
  {
    String cfgDir = Play.configuration.getProperty("virtualhosts.dir", Play.getFile("conf").getAbsolutePath());
    config.loadVirtualHostRegistrations(cfgDir);
    // Create an InotifyEventListener instance
    if (iNotifyListener == null) {
      iNotifyListener = new InotifyEventListener() {

        public void filesystemEventOccurred(InotifyEvent iEvent)
        {
          File virtualHostFile = new File(config.configDir, iEvent.getName());
          if (!virtualHostFile.getName().endsWith(".vhost")) {
            return;
          }

          if ((iEvent.getMask() & (Constants.IN_DELETE | Constants.IN_MOVED_FROM)) != 0) {
            Logger.info("Unregistering the virtual host from '%s'", virtualHostFile.getAbsolutePath());
            config.unregisterVirtualHost(virtualHostFile);
            return;
          }

          if (!virtualHostFile.isFile()) {
            return;
          }

          if ((iEvent.getMask() & (Constants.IN_CREATE | Constants.IN_MOVED_TO)) != 0) {
            Logger.info("Registering new virtual host from '%s'", virtualHostFile.getAbsolutePath());
            config.registerVirtualHost(virtualHostFile);
          } else if ((iEvent.getMask() & Constants.IN_MODIFY) != 0) {
            Logger.info("Updating virtual host registration from '%s'", virtualHostFile.getAbsolutePath());
            config.registerVirtualHost(virtualHostFile);
          }
        }

        public void queueFull(EventQueueFull e)
        {
          Logger.warn("inotify event queue '%s' is full!", e.getSource());
        }
      };
    }

    // register file change notifier for configuration directory
    try {
      iNotify = new Inotify();

    } catch (Throwable e) {
      Logger.error(e.getMessage());
      Logger.warn("Cannot initialize filesystem notification listener. You have to restart application if changes to virtual host registrations are made.");
    }
    if (iNotify != null) {
      iNotify.addInotifyEventListener(iNotifyListener);
      iNotifyWatch = iNotify.addWatch(config.configDir, Constants.IN_CREATE | Constants.IN_DELETE | Constants.IN_MODIFY | Constants.IN_MOVED_FROM | Constants.IN_MOVED_TO);
      Logger.info("Monitoring  '%s' for virtual host registrations", config.configDir);
    }
  }

  @Override
  public void onApplicationStop()
  {
    if (iNotify != null) {
      iNotify.removeInotifyEventListener(iNotifyListener);
      iNotify.removeWatch(iNotifyWatch);
      iNotify = null;
      Logger.info("VirtualHost registration monitor has been stopped");
    }
  }

  private void loadVirtualHostRegistrations(String pConfigDir)
  {
    // Clear current registrations
    synchronized (this) {
      vHostMap.clear();
      fqdnMap.clear();
    }

    configDir = pConfigDir;
    // reload existing registrations
    File cfgDir = new File(configDir);
    File[] cfgFiles = cfgDir.listFiles(new FileFilter() {

      public boolean accept(File file)
      {
        return file.getName().endsWith(".vhost") && file.isFile();
      }

    });
    if (cfgFiles == null) return;
    for (int i = 0; i < cfgFiles.length; i++) {
      Logger.info("Registering new virtual host from '%s'", cfgFiles[i].getAbsolutePath());
      registerVirtualHost(cfgFiles[i]);
    }
  }

  private void registerVirtualHost(File pVHostFile)
  {
    // Load virtual host registration file
    Properties vHostConfig = new Properties();
    String vHostFilename = pVHostFile.getAbsolutePath();
    boolean fileExists = pVHostFile.exists();
    try {
      if (fileExists) vHostConfig.load(new FileInputStream(pVHostFile));
    } catch (FileNotFoundException e) {
      Logger.error("File '%s' vanished. Registration processing is cancelled.", vHostFilename);
      return;
    } catch (IOException e) {
      Logger.info("Unexpected error occured while procesing virtual host registration from '%s'. Error message was: %s", pVHostFile.getName(), e.getMessage());
      return;
    }

    // Remove previous registration
    unregisterVirtualHost(vHostFilename);

    // Add new registration (if there is new)
    if (vHostConfig.size() == 0) return;

    String home = (String) vHostConfig.get("home", null);
    if (home == null) {
      Logger.error("Virtual host must have home directory specified. Ignoring registration file '%s'", vHostFilename);
      return;
    }

    final List<String> fqdns = new ArrayList<String>();
    for (String fqdn : ((String) vHostConfig.get("fqdns", "")).split(",")) {
      if (fqdnMap.containsKey(fqdn)) {
        Logger.warn("'%s' is already in use by another virtual host. FQDN registration skipped.", fqdn);
      } else {
        fqdns.add(fqdn);
      }
    }
    if (fqdns.size() == 0) {
      Logger.error("All fqdns for this virtual host are aleady registered. Ignoring registration file '%s'", vHostFilename);
      return;
    }

    String driver = vHostConfig.get("db.driver");
    try {
      if (driver != null) Class.forName(driver);
    } catch (ClassNotFoundException e) {
      Logger.error("Database driver not found(%s). Virtual host in '%s' not loaded", driver, home);
      return;
    }

    VirtualHost host;

    final String dbUrl = (String) vHostConfig.get("db.url", null);
    if (dbUrl == null) {
      host = new VirtualHost(home, fqdns, vHostConfig, null);
    } else {
      System.setProperty("com.mchange.v2.log.MLog", "com.mchange.v2.log.FallbackMLog");
      System.setProperty("com.mchange.v2.log.FallbackMLog.DEFAULT_CUTOFF_LEVEL", "OFF");
      final ComboPooledDataSource cpds = new ComboPooledDataSource();
      cpds.setJdbcUrl("jdbc:" + dbUrl);
      cpds.setUser((String) vHostConfig.get("db.user", ""));
      cpds.setPassword((String) vHostConfig.get("db.pass", ""));
      cpds.setCheckoutTimeout(Integer.parseInt(vHostConfig.get("db.pool.timeout", "5000")));
      cpds.setMaxPoolSize(Integer.parseInt(vHostConfig.get("db.pool.maxSize", "5")));
      cpds.setMinPoolSize(Integer.parseInt(vHostConfig.get("db.pool.minSize", "1")));
      cpds.setAutoCommitOnClose(true);
      cpds.setAcquireRetryAttempts(1);
      cpds.setAcquireRetryDelay(0);
      host = new VirtualHost(home, fqdns, vHostConfig, new VirtualHostDataSource(cpds));
    }

    synchronized (this) {
      vHostMap.put(vHostFilename, host);
      for (String fqdn : host.getFqdns())
        fqdnMap.put(fqdn, host);
    }
  }

  private void unregisterVirtualHost(File pVHostFile)
  {
    String vHostFilename = pVHostFile.getAbsolutePath();
    unregisterVirtualHost(vHostFilename);
  }

  private void unregisterVirtualHost(String pVHostFilename)
  {
    VirtualHost vHost = vHostMap.get(pVHostFilename);
    if (vHost != null) {
      synchronized (this) {
        for (String fqdn : vHost.getFqdns())
          fqdnMap.remove(fqdn);
        vHostMap.remove(pVHostFilename);
      }
      for (VirtualHostListener vhl : vHost.listeners) vhl.virtualHostUnloaded(vHost);
    }
  }

}
